<?php
/**
 * Reviews API Endpoints
 */

require_once __DIR__ . '/../cors.php';
require_once __DIR__ . '/../config.php';
require_once __DIR__ . '/../database.php';
require_once __DIR__ . '/../utils.php';

$method = $_SERVER['REQUEST_METHOD'];
$db = Database::getInstance()->getConnection();

// ============================================
// GET /api/reviews - Get reviews
// ============================================
if ($method === 'GET' && !preg_match('/\/\w+$/', $_SERVER['REQUEST_URI'])) {
    $userData = requireAuth();
    
    $mechanicId = $_GET['mechanicId'] ?? null;
    $requestId = $_GET['requestId'] ?? null;
    
    $sql = "SELECT r.*, 
            u.name as client_name,
            m.name as mechanic_name
            FROM reviews r
            LEFT JOIN users u ON r.client_id = u.id
            LEFT JOIN users m ON r.mechanic_id = m.id
            WHERE 1=1";
    
    $params = [];
    
    if ($mechanicId) {
        $sql .= " AND r.mechanic_id = ?";
        $params[] = $mechanicId;
    }
    
    if ($requestId) {
        $sql .= " AND r.request_id = ?";
        $params[] = $requestId;
    }
    
    $sql .= " ORDER BY r.created_at DESC";
    
    $stmt = $db->prepare($sql);
    $stmt->execute($params);
    $reviews = $stmt->fetchAll();
    
    sendResponse(true, 'Reviews retrieved', ['reviews' => $reviews]);
}

// ============================================
// GET /api/reviews/:id - Get review by ID
// ============================================
if ($method === 'GET' && preg_match('/\/([a-zA-Z0-9-]+)$/', $_SERVER['REQUEST_URI'], $matches)) {
    $userData = requireAuth();
    $reviewId = $matches[1];
    
    $stmt = $db->prepare("
        SELECT r.*, 
            u.name as client_name,
            m.name as mechanic_name
        FROM reviews r
        LEFT JOIN users u ON r.client_id = u.id
        LEFT JOIN users m ON r.mechanic_id = m.id
        WHERE r.id = ?
    ");
    $stmt->execute([$reviewId]);
    $review = $stmt->fetch();
    
    if (!$review) {
        sendResponse(false, 'Review not found', null, 404);
    }
    
    sendResponse(true, 'Review retrieved', ['review' => $review]);
}

// ============================================
// POST /api/reviews - Create new review
// ============================================
if ($method === 'POST') {
    $userData = requireAuth();
    
    // Only clients can create reviews
    if ($userData['userType'] !== 'client') {
        sendResponse(false, 'Only clients can create reviews', null, 403);
    }
    
    $input = getJsonInput();
    
    // Validate required fields
    if (empty($input['requestId']) || empty($input['mechanicId']) || 
        !isset($input['rating']) || empty($input['comment'])) {
        sendResponse(false, 'All fields are required', null, 400);
    }
    
    // Validate rating
    $rating = intval($input['rating']);
    if ($rating < 1 || $rating > 5) {
        sendResponse(false, 'Rating must be between 1 and 5', null, 400);
    }
    
    // Verify request exists and is completed
    $stmt = $db->prepare("SELECT * FROM breakdown_requests WHERE id = ? AND client_id = ? AND status = 'completed'");
    $stmt->execute([$input['requestId'], $userData['userId']]);
    $request = $stmt->fetch();
    
    if (!$request) {
        sendResponse(false, 'Request not found or not completed', null, 404);
    }
    
    // Check if review already exists
    $stmt = $db->prepare("SELECT id FROM reviews WHERE request_id = ?");
    $stmt->execute([$input['requestId']]);
    if ($stmt->fetch()) {
        sendResponse(false, 'Review already exists for this request', null, 409);
    }
    
    $reviewId = generateUUID();
    
    $stmt = $db->prepare("
        INSERT INTO reviews (id, request_id, client_id, mechanic_id, rating, comment, created_at, updated_at)
        VALUES (?, ?, ?, ?, ?, ?, NOW(), NOW())
    ");
    
    $stmt->execute([
        $reviewId,
        $input['requestId'],
        $userData['userId'],
        $input['mechanicId'],
        $rating,
        sanitizeInput($input['comment'])
    ]);
    
    // Get created review
    $stmt = $db->prepare("SELECT * FROM reviews WHERE id = ?");
    $stmt->execute([$reviewId]);
    $review = $stmt->fetch();
    
    sendResponse(true, 'Review created', ['review' => $review], 201);
}

// ============================================
// PUT /api/reviews/:id - Update review
// ============================================
if ($method === 'PUT' && preg_match('/\/([a-zA-Z0-9-]+)$/', $_SERVER['REQUEST_URI'], $matches)) {
    $userData = requireAuth();
    $reviewId = $matches[1];
    
    // Get review
    $stmt = $db->prepare("SELECT * FROM reviews WHERE id = ?");
    $stmt->execute([$reviewId]);
    $review = $stmt->fetch();
    
    if (!$review) {
        sendResponse(false, 'Review not found', null, 404);
    }
    
    // Only the client who created the review can update it
    if ($review['client_id'] !== $userData['userId']) {
        sendResponse(false, 'Unauthorized', null, 403);
    }
    
    $input = getJsonInput();
    
    $updateFields = [];
    $params = [];
    
    if (isset($input['rating'])) {
        $rating = intval($input['rating']);
        if ($rating < 1 || $rating > 5) {
            sendResponse(false, 'Rating must be between 1 and 5', null, 400);
        }
        $updateFields[] = "rating = ?";
        $params[] = $rating;
    }
    
    if (isset($input['comment'])) {
        $updateFields[] = "comment = ?";
        $params[] = sanitizeInput($input['comment']);
    }
    
    if (empty($updateFields)) {
        sendResponse(false, 'No fields to update', null, 400);
    }
    
    $updateFields[] = "updated_at = NOW()";
    $params[] = $reviewId;
    
    $sql = "UPDATE reviews SET " . implode(', ', $updateFields) . " WHERE id = ?";
    $stmt = $db->prepare($sql);
    $stmt->execute($params);
    
    // Get updated review
    $stmt = $db->prepare("SELECT * FROM reviews WHERE id = ?");
    $stmt->execute([$reviewId]);
    $review = $stmt->fetch();
    
    sendResponse(true, 'Review updated', ['review' => $review]);
}

// ============================================
// DELETE /api/reviews/:id - Delete review
// ============================================
if ($method === 'DELETE' && preg_match('/\/([a-zA-Z0-9-]+)$/', $_SERVER['REQUEST_URI'], $matches)) {
    $userData = requireAuth();
    $reviewId = $matches[1];
    
    // Get review
    $stmt = $db->prepare("SELECT * FROM reviews WHERE id = ?");
    $stmt->execute([$reviewId]);
    $review = $stmt->fetch();
    
    if (!$review) {
        sendResponse(false, 'Review not found', null, 404);
    }
    
    // Only the client who created the review can delete it
    if ($review['client_id'] !== $userData['userId']) {
        sendResponse(false, 'Unauthorized', null, 403);
    }
    
    $stmt = $db->prepare("DELETE FROM reviews WHERE id = ?");
    $stmt->execute([$reviewId]);
    
    sendResponse(true, 'Review deleted');
}

// Default response
sendResponse(false, 'Endpoint not found', null, 404);
