<?php
/**
 * Users API Endpoints
 */

require_once __DIR__ . '/../cors.php';
require_once __DIR__ . '/../config.php';
require_once __DIR__ . '/../database.php';
require_once __DIR__ . '/../utils.php';

$method = $_SERVER['REQUEST_METHOD'];
$db = Database::getInstance()->getConnection();

// ============================================
// GET /api/users - Get users (mechanics/clients)
// ============================================
if ($method === 'GET' && !preg_match('/\/\w+$/', $_SERVER['REQUEST_URI'])) {
    $userData = requireAuth();
    
    $userType = $_GET['userType'] ?? null;
    $isOnline = $_GET['isOnline'] ?? null;
    
    $sql = "SELECT id, name, email, phone, user_type, is_online, profile_picture, latitude, longitude, created_at FROM users WHERE 1=1";
    $params = [];
    
    if ($userType) {
        $sql .= " AND user_type = ?";
        $params[] = $userType;
    }
    
    if ($isOnline !== null) {
        $sql .= " AND is_online = ?";
        $params[] = $isOnline;
    }
    
    $stmt = $db->prepare($sql);
    $stmt->execute($params);
    $users = $stmt->fetchAll();
    
    sendResponse(true, 'Users retrieved', ['users' => $users]);
}

// ============================================
// GET /api/users/:id - Get user by ID
// ============================================
if ($method === 'GET' && preg_match('/\/([a-zA-Z0-9-]+)$/', $_SERVER['REQUEST_URI'], $matches)) {
    $userData = requireAuth();
    $userId = $matches[1];
    
    $stmt = $db->prepare("SELECT id, name, email, phone, user_type, is_online, profile_picture, address, latitude, longitude, created_at FROM users WHERE id = ?");
    $stmt->execute([$userId]);
    $user = $stmt->fetch();
    
    if (!$user) {
        sendResponse(false, 'User not found', null, 404);
    }
    
    // Get average rating for mechanics
    if ($user['user_type'] === 'mechanic') {
        $stmt = $db->prepare("SELECT AVG(rating) as average_rating, COUNT(*) as total_reviews FROM reviews WHERE mechanic_id = ?");
        $stmt->execute([$userId]);
        $ratings = $stmt->fetch();
        $user['average_rating'] = round($ratings['average_rating'], 1);
        $user['total_reviews'] = $ratings['total_reviews'];
    }
    
    sendResponse(true, 'User retrieved', ['user' => $user]);
}

// ============================================
// PUT /api/users/:id - Update user
// ============================================
if ($method === 'PUT' && preg_match('/\/([a-zA-Z0-9-]+)$/', $_SERVER['REQUEST_URI'], $matches)) {
    $userData = requireAuth();
    $userId = $matches[1];
    
    // Check if user can update this profile
    if ($userData['userId'] !== $userId) {
        sendResponse(false, 'Unauthorized', null, 403);
    }
    
    $input = getJsonInput();
    
    $updateFields = [];
    $params = [];
    
    if (isset($input['name'])) {
        $updateFields[] = "name = ?";
        $params[] = sanitizeInput($input['name']);
    }
    
    if (isset($input['phone'])) {
        $updateFields[] = "phone = ?";
        $params[] = sanitizeInput($input['phone']);
    }
    
    if (isset($input['address'])) {
        $updateFields[] = "address = ?";
        $params[] = sanitizeInput($input['address']);
    }
    
    if (isset($input['latitude'])) {
        $updateFields[] = "latitude = ?";
        $params[] = $input['latitude'];
    }
    
    if (isset($input['longitude'])) {
        $updateFields[] = "longitude = ?";
        $params[] = $input['longitude'];
    }
    
    if (isset($input['isOnline'])) {
        $updateFields[] = "is_online = ?";
        $params[] = $input['isOnline'] ? 1 : 0;
    }
    
    if (empty($updateFields)) {
        sendResponse(false, 'No fields to update', null, 400);
    }
    
    $updateFields[] = "updated_at = NOW()";
    $params[] = $userId;
    
    $sql = "UPDATE users SET " . implode(', ', $updateFields) . " WHERE id = ?";
    $stmt = $db->prepare($sql);
    $stmt->execute($params);
    
    // Get updated user
    $stmt = $db->prepare("SELECT id, name, email, phone, user_type, is_online, profile_picture, address, latitude, longitude, updated_at FROM users WHERE id = ?");
    $stmt->execute([$userId]);
    $user = $stmt->fetch();
    
    sendResponse(true, 'User updated', ['user' => $user]);
}

// ============================================
// GET /api/users/nearby - Get nearby mechanics
// ============================================
if ($method === 'GET' && strpos($_SERVER['REQUEST_URI'], '/nearby') !== false) {
    $userData = requireAuth();
    
    $latitude = $_GET['latitude'] ?? null;
    $longitude = $_GET['longitude'] ?? null;
    $radius = $_GET['radius'] ?? 50; // km
    
    if (!$latitude || !$longitude) {
        sendResponse(false, 'Latitude and longitude are required', null, 400);
    }
    
    // Get all online mechanics
    $stmt = $db->prepare("
        SELECT id, name, email, phone, user_type, is_online, profile_picture, latitude, longitude
        FROM users 
        WHERE user_type = 'mechanic' 
        AND is_online = 1 
        AND latitude IS NOT NULL 
        AND longitude IS NOT NULL
    ");
    $stmt->execute();
    $mechanics = $stmt->fetchAll();
    
    // Filter by distance
    $nearbyMechanics = [];
    foreach ($mechanics as $mechanic) {
        $distance = calculateDistance(
            $latitude, 
            $longitude, 
            $mechanic['latitude'], 
            $mechanic['longitude']
        );
        
        if ($distance <= $radius) {
            $mechanic['distance'] = round($distance, 2);
            
            // Get average rating
            $stmt = $db->prepare("SELECT AVG(rating) as average_rating, COUNT(*) as total_reviews FROM reviews WHERE mechanic_id = ?");
            $stmt->execute([$mechanic['id']]);
            $ratings = $stmt->fetch();
            $mechanic['average_rating'] = round($ratings['average_rating'], 1);
            $mechanic['total_reviews'] = $ratings['total_reviews'];
            
            $nearbyMechanics[] = $mechanic;
        }
    }
    
    // Sort by distance
    usort($nearbyMechanics, function($a, $b) {
        return $a['distance'] <=> $b['distance'];
    });
    
    sendResponse(true, 'Nearby mechanics retrieved', ['mechanics' => $nearbyMechanics]);
}

// Default response
sendResponse(false, 'Endpoint not found', null, 404);
