<?php
/**
 * Email Service using PHPMailer
 * 
 * Handles sending emails via SMTP
 */

use PHPMailer\PHPMailer\PHPMailer;
use PHPMailer\PHPMailer\Exception;

class EmailService {
    private $mailer;

    public function __construct() {
        $this->mailer = new PHPMailer(true);
        $this->configure();
    }

    private function configure() {
        try {
            // Server settings
            $this->mailer->isSMTP();
            $this->mailer->Host       = SMTP_HOST;
            $this->mailer->SMTPAuth   = true;
            $this->mailer->Username   = SMTP_USERNAME;
            $this->mailer->Password   = SMTP_PASSWORD;
            $this->mailer->SMTPSecure = PHPMailer::ENCRYPTION_SMTPS;
            $this->mailer->Port       = SMTP_PORT;

            // Sender
            $this->mailer->setFrom(SMTP_FROM_EMAIL, SMTP_FROM_NAME);
            $this->mailer->CharSet = 'UTF-8';
        } catch (Exception $e) {
            error_log("Email configuration error: " . $e->getMessage());
        }
    }

    /**
     * Send a simple email
     */
    public function sendEmail($to, $subject, $body, $isHTML = true) {
        try {
            $this->mailer->clearAddresses();
            $this->mailer->addAddress($to);
            $this->mailer->isHTML($isHTML);
            $this->mailer->Subject = $subject;
            $this->mailer->Body    = $body;

            if ($this->mailer->send()) {
                return ['success' => true, 'message' => 'Email sent successfully'];
            }
            return ['success' => false, 'message' => 'Email failed to send'];
        } catch (Exception $e) {
            error_log("Email sending error: " . $e->getMessage());
            return ['success' => false, 'message' => $e->getMessage()];
        }
    }

    /**
     * Send welcome email to new users
     */
    public function sendWelcomeEmail($email, $name, $userType) {
        $subject = "Bienvenue sur MECASOS";
        $body = "
            <html>
            <head>
                <style>
                    body { font-family: Arial, sans-serif; }
                    .container { max-width: 600px; margin: 0 auto; padding: 20px; }
                    .header { background-color: #E1A004; color: white; padding: 20px; text-align: center; }
                    .content { padding: 20px; background-color: #f4f4f4; }
                    .footer { text-align: center; padding: 10px; font-size: 12px; color: #666; }
                </style>
            </head>
            <body>
                <div class='container'>
                    <div class='header'>
                        <h1>MECASOS</h1>
                    </div>
                    <div class='content'>
                        <h2>Bonjour $name,</h2>
                        <p>Bienvenue sur MECASOS, votre plateforme de dépannage automobile premium.</p>
                        " . ($userType === 'client' 
                            ? "<p>Vous pouvez maintenant demander un dépannage en quelques clics.</p>" 
                            : "<p>Vous pouvez maintenant accepter des demandes de dépannage et augmenter vos revenus.</p>") . "
                        <p>Merci de nous avoir rejoint !</p>
                    </div>
                    <div class='footer'>
                        <p>&copy; 2024 MECASOS. Tous droits réservés.</p>
                    </div>
                </div>
            </body>
            </html>
        ";
        
        return $this->sendEmail($email, $subject, $body);
    }

    /**
     * Send password reset email
     */
    public function sendPasswordResetEmail($email, $resetToken) {
        $resetLink = APP_URL . "/reset-password?token=" . $resetToken;
        $subject = "Réinitialisation de votre mot de passe MECASOS";
        $body = "
            <html>
            <head>
                <style>
                    body { font-family: Arial, sans-serif; }
                    .container { max-width: 600px; margin: 0 auto; padding: 20px; }
                    .header { background-color: #1D1D1B; color: #E1A004; padding: 20px; text-align: center; }
                    .content { padding: 20px; background-color: #f4f4f4; }
                    .button { background-color: #E1A004; color: white; padding: 10px 20px; text-decoration: none; display: inline-block; margin: 20px 0; border-radius: 5px; }
                    .footer { text-align: center; padding: 10px; font-size: 12px; color: #666; }
                </style>
            </head>
            <body>
                <div class='container'>
                    <div class='header'>
                        <h1>MECASOS</h1>
                    </div>
                    <div class='content'>
                        <h2>Réinitialisation de mot de passe</h2>
                        <p>Vous avez demandé la réinitialisation de votre mot de passe.</p>
                        <p>Cliquez sur le bouton ci-dessous pour réinitialiser votre mot de passe :</p>
                        <a href='$resetLink' class='button'>Réinitialiser mon mot de passe</a>
                        <p>Ce lien expirera dans 1 heure.</p>
                        <p>Si vous n'avez pas demandé cette réinitialisation, ignorez cet email.</p>
                    </div>
                    <div class='footer'>
                        <p>&copy; 2024 MECASOS. Tous droits réservés.</p>
                    </div>
                </div>
            </body>
            </html>
        ";
        
        return $this->sendEmail($email, $subject, $body);
    }

    /**
     * Send request notification to mechanic
     */
    public function sendRequestNotification($mechanicEmail, $mechanicName, $requestDetails) {
        $subject = "Nouvelle demande de dépannage";
        $body = "
            <html>
            <head>
                <style>
                    body { font-family: Arial, sans-serif; }
                    .container { max-width: 600px; margin: 0 auto; padding: 20px; }
                    .header { background-color: #E1A004; color: white; padding: 20px; text-align: center; }
                    .content { padding: 20px; background-color: #f4f4f4; }
                    .footer { text-align: center; padding: 10px; font-size: 12px; color: #666; }
                </style>
            </head>
            <body>
                <div class='container'>
                    <div class='header'>
                        <h1>MECASOS</h1>
                    </div>
                    <div class='content'>
                        <h2>Bonjour $mechanicName,</h2>
                        <p>Une nouvelle demande de dépannage est disponible :</p>
                        <ul>
                            <li><strong>Type:</strong> {$requestDetails['type']}</li>
                            <li><strong>Adresse:</strong> {$requestDetails['address']}</li>
                            <li><strong>Description:</strong> {$requestDetails['description']}</li>
                        </ul>
                        <p>Connectez-vous à l'application pour accepter cette demande.</p>
                    </div>
                    <div class='footer'>
                        <p>&copy; 2024 MECASOS. Tous droits réservés.</p>
                    </div>
                </div>
            </body>
            </html>
        ";
        
        return $this->sendEmail($mechanicEmail, $subject, $body);
    }
}
