<?php
$pageTitle = 'Historique des Transactions';
$breadcrumbs = [
    ['label' => 'Dashboard', 'url' => '../index.php'],
    ['label' => 'Finances', 'url' => 'overview.php'],
    ['label' => 'Transactions', 'url' => '#']
];

require_once __DIR__ . '/../../config/constants.php';
require_once __DIR__ . '/../../config/functions.php';

// Charger les données mock
$interventionsData = json_decode(file_get_contents(__DIR__ . '/../../mockdata/interventions.json'), true);
$usersData = json_decode(file_get_contents(__DIR__ . '/../../mockdata/users.json'), true);
$mechanicsData = json_decode(file_get_contents(__DIR__ . '/../../mockdata/mechanics.json'), true);

$interventions = $interventionsData['interventions'];

// Générer des transactions mock basées sur les interventions
$transactions = [];
foreach($interventions as $intervention) {
    // Transaction de paiement client
    $transactions[] = [
        'id' => 'TXN-PAY-' . str_pad($intervention['id'], 3, '0', STR_PAD_LEFT),
        'type' => 'payment',
        'description' => 'Paiement intervention ' . $intervention['id'],
        'client' => $intervention['client'],
        'mechanic' => $intervention['mechanic'],
        'amount' => $intervention['amount'],
        'method' => $intervention['payment_method'],
        'status' => $intervention['payment_status'],
        'date' => $intervention['created_at'],
        'reference' => $intervention['id']
    ];

    // Transaction de commission mécanicien (si payée)
    if($intervention['status'] === 'completed') {
        $transactions[] = [
            'id' => 'TXN-COM-' . str_pad($intervention['id'], 3, '0', STR_PAD_LEFT),
            'type' => 'commission',
            'description' => 'Commission intervention ' . $intervention['id'],
            'client' => $intervention['client'],
            'mechanic' => $intervention['mechanic'],
            'amount' => $intervention['commission'],
            'method' => 'bank_transfer',
            'status' => 'completed',
            'date' => $intervention['completed_at'] ?? $intervention['created_at'],
            'reference' => $intervention['id']
        ];
    }
}

// Trier par date décroissante
usort($transactions, function($a, $b) {
    return strtotime($b['date']) - strtotime($a['date']);
});

// Filtres
$typeFilter = $_GET['type'] ?? 'all';
$statusFilter = $_GET['status'] ?? 'all';
$dateFrom = $_GET['date_from'] ?? '';
$dateTo = $_GET['date_to'] ?? '';
$amountMin = $_GET['amount_min'] ?? '';
$amountMax = $_GET['amount_max'] ?? '';

// Appliquer les filtres
$filteredTransactions = array_filter($transactions, function($transaction) use ($typeFilter, $statusFilter, $dateFrom, $dateTo, $amountMin, $amountMax) {
    if ($typeFilter !== 'all' && $transaction['type'] !== $typeFilter) return false;
    if ($statusFilter !== 'all' && $transaction['status'] !== $statusFilter) return false;
    if ($dateFrom && strtotime($transaction['date']) < strtotime($dateFrom)) return false;
    if ($dateTo && strtotime($transaction['date']) > strtotime($dateTo)) return false;
    if ($amountMin && $transaction['amount'] < floatval($amountMin)) return false;
    if ($amountMax && $transaction['amount'] > floatval($amountMax)) return false;
    return true;
});

// Pagination
$page = $_GET['page'] ?? 1;
$perPage = 25;
$totalPages = ceil(count($filteredTransactions) / $perPage);
$offset = ($page - 1) * $perPage;
$pagedTransactions = array_slice($filteredTransactions, $offset, $perPage);

// Statistiques
$totalTransactions = count($filteredTransactions);
$totalAmount = array_sum(array_column($filteredTransactions, 'amount'));
$completedTransactions = count(array_filter($filteredTransactions, fn($t) => $t['status'] === 'completed'));
$pendingTransactions = count(array_filter($filteredTransactions, fn($t) => $t['status'] === 'pending'));
?>

<!-- Page Header -->
<div class="page-header">
    <div class="page-header-content">
        <h1 class="page-title">
            <i class="bi bi-receipt"></i>
            Historique des Transactions
        </h1>
        <p class="page-subtitle">Suivi détaillé de tous les mouvements financiers</p>
    </div>
    <div class="page-header-actions">
        <div class="stats-summary">
            <div class="stat-item">
                <span class="stat-value"><?= $totalTransactions ?></span>
                <span class="stat-label">Transactions</span>
            </div>
            <div class="stat-item">
                <span class="stat-value text-gold"><?= formatCurrency($totalAmount) ?></span>
                <span class="stat-label">Total</span>
            </div>
        </div>
        <button class="btn btn-outline-primary btn-icon-text" onclick="exportCSV()">
            <i class="bi bi-file-earmark-spreadsheet"></i>
            <span>CSV</span>
        </button>
        <button class="btn btn-outline-primary btn-icon-text" onclick="exportPDF()">
            <i class="bi bi-file-earmark-pdf"></i>
            <span>PDF</span>
        </button>
    </div>
</div>

<!-- Filters Bar -->
<div class="dashboard-card mb-4">
    <div class="card-body">
        <form method="GET" class="filters-form">
            <div class="row g-3">
                <div class="col-md-2">
                    <label class="form-label">Type</label>
                    <select name="type" class="form-select">
                        <option value="all" <?= $typeFilter === 'all' ? 'selected' : '' ?>>Tous types</option>
                        <option value="payment" <?= $typeFilter === 'payment' ? 'selected' : '' ?>>Paiement</option>
                        <option value="commission" <?= $typeFilter === 'commission' ? 'selected' : '' ?>>Commission</option>
                        <option value="refund" <?= $typeFilter === 'refund' ? 'selected' : '' ?>>Remboursement</option>
                    </select>
                </div>

                <div class="col-md-2">
                    <label class="form-label">Statut</label>
                    <select name="status" class="form-select">
                        <option value="all" <?= $statusFilter === 'all' ? 'selected' : '' ?>>Tous statuts</option>
                        <option value="completed" <?= $statusFilter === 'completed' ? 'selected' : '' ?>>Terminé</option>
                        <option value="pending" <?= $statusFilter === 'pending' ? 'selected' : '' ?>>En attente</option>
                        <option value="failed" <?= $statusFilter === 'failed' ? 'selected' : '' ?>>Échec</option>
                        <option value="cancelled" <?= $statusFilter === 'cancelled' ? 'selected' : '' ?>>Annulé</option>
                    </select>
                </div>

                <div class="col-md-2">
                    <label class="form-label">Date début</label>
                    <input type="date" name="date_from" class="form-control" value="<?= $dateFrom ?>">
                </div>

                <div class="col-md-2">
                    <label class="form-label">Date fin</label>
                    <input type="date" name="date_to" class="form-control" value="<?= $dateTo ?>">
                </div>

                <div class="col-md-2">
                    <label class="form-label">Montant min</label>
                    <input type="number" name="amount_min" class="form-control" placeholder="0" value="<?= $amountMin ?>">
                </div>

                <div class="col-md-2">
                    <label class="form-label">Montant max</label>
                    <input type="number" name="amount_max" class="form-control" placeholder="∞" value="<?= $amountMax ?>">
                </div>
            </div>

            <div class="row mt-3">
                <div class="col-12">
                    <div class="d-flex gap-2">
                        <button type="submit" class="btn btn-primary">
                            <i class="bi bi-search"></i>
                            Filtrer
                        </button>
                        <a href="transactions.php" class="btn btn-outline-secondary">
                            <i class="bi bi-x"></i>
                            Réinitialiser
                        </a>
                        <div class="ms-auto">
                            <small class="text-muted">
                                <?= count($filteredTransactions) ?> transactions trouvées
                            </small>
                        </div>
                    </div>
                </div>
            </div>
        </form>
    </div>
</div>

<!-- Data Table -->
<div class="dashboard-card">
    <div class="card-header">
        <div class="card-header-content">
            <h5 class="card-title">
                <i class="bi bi-table"></i>
                Transactions
            </h5>
            <p class="card-subtitle">
                <?= $completedTransactions ?> terminées • <?= $pendingTransactions ?> en attente
            </p>
        </div>
        <div class="card-header-actions">
            <div class="btn-group" role="group">
                <button type="button" class="btn btn-sm btn-outline-secondary" id="selectAllBtn">
                    <i class="bi bi-check-all"></i>
                    Tout sélectionner
                </button>
                <button type="button" class="btn btn-sm btn-outline-secondary" id="bulkActionsBtn" disabled>
                    <i class="bi bi-three-dots"></i>
                    Actions groupées
                </button>
            </div>
        </div>
    </div>
    <div class="card-body p-0">
        <div class="table-responsive">
            <table class="data-table">
                <thead>
                    <tr>
                        <th width="40">
                            <input type="checkbox" class="form-check-input" id="selectAll">
                        </th>
                        <th>ID Transaction</th>
                        <th>Type</th>
                        <th>Client/Mécanicien</th>
                        <th>Description</th>
                        <th>Montant</th>
                        <th>Méthode</th>
                        <th>Statut</th>
                        <th>Date</th>
                        <th width="120">Actions</th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach($pagedTransactions as $transaction): ?>
                    <tr>
                        <td>
                            <input type="checkbox" class="form-check-input row-checkbox" value="<?= $transaction['id'] ?>">
                        </td>
                        <td>
                            <div class="transaction-id">
                                <span class="badge bg-dark"><?= $transaction['id'] ?></span>
                            </div>
                        </td>
                        <td>
                            <span class="transaction-type type-<?= $transaction['type'] ?>">
                                <i class="bi bi-<?= getTransactionIcon($transaction['type']) ?>"></i>
                                <?= ucfirst($transaction['type']) ?>
                            </span>
                        </td>
                        <td>
                            <div class="user-cell">
                                <?php if($transaction['type'] === 'payment'): ?>
                                    <img src="<?= getDefaultAvatar($transaction['client']) ?>" alt="" class="user-avatar-sm">
                                    <div class="user-info">
                                        <span class="user-name"><?= e($transaction['client']) ?></span>
                                        <small class="text-muted">Client</small>
                                    </div>
                                <?php else: ?>
                                    <img src="<?= getDefaultAvatar($transaction['mechanic']) ?>" alt="" class="user-avatar-sm">
                                    <div class="user-info">
                                        <span class="user-name"><?= e($transaction['mechanic']) ?></span>
                                        <small class="text-muted">Mécanicien</small>
                                    </div>
                                <?php endif; ?>
                            </div>
                        </td>
                        <td>
                            <div class="transaction-description">
                                <span class="description-text"><?= e($transaction['description']) ?></span>
                                <?php if($transaction['reference']): ?>
                                <br><small class="text-muted">Ref: <?= $transaction['reference'] ?></small>
                                <?php endif; ?>
                            </div>
                        </td>
                        <td>
                            <span class="transaction-amount <?= $transaction['type'] === 'refund' ? 'text-danger' : 'text-gold' ?>">
                                <?php if($transaction['type'] === 'refund'): ?>
                                    -<?= formatCurrency($transaction['amount']) ?>
                                <?php else: ?>
                                    <?= formatCurrency($transaction['amount']) ?>
                                <?php endif; ?>
                            </span>
                        </td>
                        <td>
                            <span class="payment-method">
                                <i class="bi bi-<?= getPaymentMethodIcon($transaction['method']) ?>"></i>
                                <?= getPaymentMethodLabel($transaction['method']) ?>
                            </span>
                        </td>
                        <td>
                            <?= getStatusBadge($transaction['status']) ?>
                        </td>
                        <td>
                            <small class="text-muted">
                                <i class="bi bi-calendar"></i>
                                <?= formatDate($transaction['date'], 'd/m/Y H:i') ?>
                            </small>
                        </td>
                        <td>
                            <div class="btn-group" role="group">
                                <button class="btn btn-sm btn-icon btn-outline-primary" title="Voir détails"
                                        onclick="viewTransactionDetails('<?= $transaction['id'] ?>')">
                                    <i class="bi bi-eye"></i>
                                </button>
                                <button class="btn btn-sm btn-icon btn-outline-secondary" title="Télécharger reçu"
                                        onclick="downloadReceipt('<?= $transaction['id'] ?>')">
                                    <i class="bi bi-download"></i>
                                </button>
                                <div class="dropdown">
                                    <button class="btn btn-sm btn-icon btn-outline-secondary dropdown-toggle"
                                            type="button" data-bs-toggle="dropdown">
                                        <i class="bi bi-three-dots"></i>
                                    </button>
                                    <ul class="dropdown-menu">
                                        <?php if($transaction['status'] === 'pending'): ?>
                                        <li><a class="dropdown-item" href="#" onclick="processTransaction('<?= $transaction['id'] ?>')">
                                            <i class="bi bi-check-circle"></i> Traiter
                                        </a></li>
                                        <?php endif; ?>
                                        <?php if($transaction['type'] === 'payment' && $transaction['status'] === 'completed'): ?>
                                        <li><a class="dropdown-item" href="#" onclick="refundTransaction('<?= $transaction['id'] ?>')">
                                            <i class="bi bi-arrow-counterclockwise"></i> Rembourser
                                        </a></li>
                                        <?php endif; ?>
                                        <li><hr class="dropdown-divider"></li>
                                        <li><a class="dropdown-item" href="#" onclick="contactUser('<?= $transaction['id'] ?>')">
                                            <i class="bi bi-chat"></i> Contacter l'utilisateur
                                        </a></li>
                                    </ul>
                                </div>
                            </div>
                        </td>
                    </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
        </div>
    </div>

    <!-- Pagination -->
    <?php if($totalPages > 1): ?>
    <div class="card-footer">
        <nav aria-label="Pagination">
            <ul class="pagination justify-content-center">
                <li class="page-item <?= $page <= 1 ? 'disabled' : '' ?>">
                    <a class="page-link" href="?page=<?= $page - 1 ?>&type=<?= $typeFilter ?>&status=<?= $statusFilter ?>&date_from=<?= $dateFrom ?>&date_to=<?= $dateTo ?>&amount_min=<?= $amountMin ?>&amount_max=<?= $amountMax ?>">
                        <i class="bi bi-chevron-left"></i>
                    </a>
                </li>

                <?php for($i = max(1, $page - 2); $i <= min($totalPages, $page + 2); $i++): ?>
                <li class="page-item <?= $i === $page ? 'active' : '' ?>">
                    <a class="page-link" href="?page=<?= $i ?>&type=<?= $typeFilter ?>&status=<?= $statusFilter ?>&date_from=<?= $dateFrom ?>&date_to=<?= $dateTo ?>&amount_min=<?= $amountMin ?>&amount_max=<?= $amountMax ?>"><?= $i ?></a>
                </li>
                <?php endfor; ?>

                <li class="page-item <?= $page >= $totalPages ? 'disabled' : '' ?>">
                    <a class="page-link" href="?page=<?= $page + 1 ?>&type=<?= $typeFilter ?>&status=<?= $statusFilter ?>&date_from=<?= $dateFrom ?>&date_to=<?= $dateTo ?>&amount_min=<?= $amountMin ?>&amount_max=<?= $amountMax ?>">
                        <i class="bi bi-chevron-right"></i>
                    </a>
                </li>
            </ul>
        </nav>
    </div>
    <?php endif; ?>
</div>

<?php
function getTransactionIcon($type) {
    $icons = [
        'payment' => 'credit-card',
        'commission' => 'coin',
        'refund' => 'arrow-counterclockwise'
    ];
    return $icons[$type] ?? 'question-circle';
}

function getPaymentMethodIcon($method) {
    $icons = [
        'wallet' => 'wallet2',
        'card' => 'credit-card',
        'cash' => 'cash',
        'bank_transfer' => 'bank'
    ];
    return $icons[$method] ?? 'question-circle';
}

function getPaymentMethodLabel($method) {
    $labels = [
        'wallet' => 'Portefeuille',
        'card' => 'Carte',
        'cash' => 'Espèces',
        'bank_transfer' => 'Virement'
    ];
    return $labels[$method] ?? $method;
}
?>

<script>
// ========================================
// TRANSACTIONS FUNCTIONALITY
// ========================================
document.addEventListener('DOMContentLoaded', function() {
    // Select all functionality
    const selectAllCheckbox = document.getElementById('selectAll');
    const rowCheckboxes = document.querySelectorAll('.row-checkbox');
    const bulkActionsBtn = document.getElementById('bulkActionsBtn');
    const selectAllBtn = document.getElementById('selectAllBtn');

    selectAllCheckbox.addEventListener('change', function() {
        rowCheckboxes.forEach(checkbox => {
            checkbox.checked = this.checked;
        });
        updateBulkActions();
    });

    rowCheckboxes.forEach(checkbox => {
        checkbox.addEventListener('change', function() {
            const checkedBoxes = document.querySelectorAll('.row-checkbox:checked');
            selectAllCheckbox.checked = checkedBoxes.length === rowCheckboxes.length;
            selectAllCheckbox.indeterminate = checkedBoxes.length > 0 && checkedBoxes.length < rowCheckboxes.length;
            updateBulkActions();
        });
    });

    selectAllBtn.addEventListener('click', function() {
        const allChecked = document.querySelectorAll('.row-checkbox:checked').length === rowCheckboxes.length;
        selectAllCheckbox.checked = !allChecked;
        selectAllCheckbox.dispatchEvent(new Event('change'));
    });

    function updateBulkActions() {
        const checkedBoxes = document.querySelectorAll('.row-checkbox:checked');
        bulkActionsBtn.disabled = checkedBoxes.length === 0;
    }
});

// Functions
function viewTransactionDetails(id) {
    showNotification('Ouverture des détails de la transaction...', 'info');
    // Simulation d'ouverture de modal de détails
}

function downloadReceipt(id) {
    showNotification('Téléchargement du reçu...', 'info');
    setTimeout(() => {
        showNotification('Reçu téléchargé!', 'success');
    }, 1000);
}

function processTransaction(id) {
    showNotification('Traitement de la transaction...', 'info');
    setTimeout(() => {
        showNotification('Transaction traitée avec succès!', 'success');
    }, 1500);
}

function refundTransaction(id) {
    if(confirm('Êtes-vous sûr de vouloir rembourser cette transaction?')) {
        showNotification('Demande de remboursement envoyée...', 'warning');
        setTimeout(() => {
            showNotification('Remboursement traité!', 'success');
        }, 2000);
    }
}

function contactUser(id) {
    showNotification('Ouverture du chat utilisateur...', 'info');
}

function exportCSV() {
    showNotification('Export CSV en cours...', 'info');
    setTimeout(() => {
        showNotification('Fichier CSV généré!', 'success');
    }, 2000);
}

function exportPDF() {
    showNotification('Export PDF en cours...', 'info');
    setTimeout(() => {
        showNotification('Rapport PDF généré!', 'success');
    }, 3000);
}

function showNotification(message, type = 'info') {
    const notification = document.createElement('div');
    notification.className = `alert alert-${type} alert-dismissible fade show position-fixed`;
    notification.style.cssText = 'top: 20px; right: 20px; z-index: 9999; min-width: 300px;';
    notification.innerHTML = `
        ${message}
        <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
    `;

    document.body.appendChild(notification);

    setTimeout(() => {
        if(notification.parentNode) {
            notification.remove();
        }
    }, 5000);
}
</script>

<style>
/* Transaction Styles */
.stats-summary {
    display: flex;
    gap: 24px;
    margin-right: 24px;
}

.stat-item {
    text-align: center;
}

.stat-value {
    display: block;
    font-size: 20px;
    font-weight: 700;
    color: var(--text-primary);
}

.stat-label {
    font-size: 12px;
    color: var(--text-secondary);
    text-transform: uppercase;
    letter-spacing: 0.5px;
}

.filters-form .form-label {
    font-weight: 500;
    color: var(--text-primary);
    margin-bottom: 8px;
}

.data-table {
    width: 100%;
    margin-bottom: 0;
}

.data-table thead th {
    background: var(--card-background);
    border-bottom: 2px solid var(--border-color);
    font-weight: 600;
    color: var(--text-primary);
    padding: 16px 12px;
    text-align: left;
}

.data-table tbody td {
    padding: 16px 12px;
    border-bottom: 1px solid rgba(255, 255, 255, 0.05);
    vertical-align: middle;
}

.data-table tbody tr:hover {
    background: rgba(255, 255, 255, 0.02);
}

.transaction-id .badge {
    font-size: 11px;
    padding: 4px 8px;
}

.transaction-type {
    display: inline-flex;
    align-items: center;
    gap: 6px;
    padding: 4px 8px;
    border-radius: 4px;
    font-size: 12px;
    font-weight: 500;
}

.type-payment {
    background: rgba(16, 185, 129, 0.1);
    color: #10B981;
}

.type-commission {
    background: rgba(59, 130, 246, 0.1);
    color: #3B82F6;
}

.type-refund {
    background: rgba(239, 68, 68, 0.1);
    color: #EF4444;
}

.user-cell {
    display: flex;
    align-items: center;
    gap: 10px;
}

.user-avatar-sm {
    width: 32px;
    height: 32px;
    border-radius: 50%;
    object-fit: cover;
}

.user-info {
    display: flex;
    flex-direction: column;
}

.user-name {
    font-weight: 500;
    color: var(--text-primary);
}

.transaction-description {
    max-width: 200px;
}

.description-text {
    color: var(--text-primary);
}

.transaction-amount {
    font-weight: 600;
    font-size: 14px;
}

.payment-method {
    display: inline-flex;
    align-items: center;
    gap: 6px;
    font-size: 12px;
    color: var(--text-secondary);
}

.btn-group .btn {
    padding: 6px 8px;
}

.btn-icon {
    width: 32px;
    height: 32px;
    display: flex;
    align-items: center;
    justify-content: center;
    padding: 0;
}

.dropdown-menu {
    background: var(--card-background);
    border: 1px solid var(--border-color);
    box-shadow: 0 8px 25px rgba(0, 0, 0, 0.3);
}

.dropdown-item {
    color: var(--text-primary);
    padding: 8px 16px;
}

.dropdown-item:hover {
    background: rgba(255, 255, 255, 0.05);
}

.pagination {
    margin-bottom: 0;
}

.pagination .page-link {
    background: var(--card-background);
    border-color: var(--border-color);
    color: var(--text-primary);
    padding: 8px 12px;
}

.pagination .page-link:hover {
    background: rgba(255, 255, 255, 0.05);
    border-color: var(--primary-color);
    color: var(--primary-color);
}

.pagination .page-item.active .page-link {
    background: var(--primary-color);
    border-color: var(--primary-color);
}

/* Responsive */
@media (max-width: 768px) {
    .page-header-actions {
        flex-direction: column;
        gap: 12px;
    }

    .stats-summary {
        margin-right: 0;
        margin-bottom: 12px;
    }

    .filters-form .row > div {
        margin-bottom: 16px;
    }

    .data-table {
        font-size: 14px;
    }

    .data-table thead th,
    .data-table tbody td {
        padding: 8px 6px;
    }

    .user-cell {
        flex-direction: column;
        text-align: center;
        gap: 4px;
    }

    .transaction-description {
        max-width: none;
    }

    .btn-group {
        flex-direction: column;
        gap: 4px;
    }

    .btn-group .btn {
        width: 100%;
    }
}
</style>