<?php
$pageTitle = 'Détails de l\'Intervention';
$breadcrumbs = [
    ['label' => 'Dashboard', 'url' => '../index.php'],
    ['label' => 'Interventions', 'url' => 'list.php'],
    ['label' => 'Détails', 'url' => '#']
];

require_once __DIR__ . '/../../config/constants.php';
require_once __DIR__ . '/../../config/functions.php';

// Charger les données mock
$interventionsData = json_decode(file_get_contents(__DIR__ . '/../../mockdata/interventions.json'), true);
$usersData = json_decode(file_get_contents(__DIR__ . '/../../mockdata/users.json'), true);
$mechanicsData = json_decode(file_get_contents(__DIR__ . '/../../mockdata/mechanics.json'), true);

// Récupérer l'ID de l'intervention depuis l'URL
$interventionId = $_GET['id'] ?? 'INT-001';

// Trouver l'intervention
$intervention = null;
foreach($interventionsData['interventions'] as $int) {
    if($int['id'] === $interventionId) {
        $intervention = $int;
        break;
    }
}

// Si l'intervention n'existe pas, rediriger
if(!$intervention) {
    header('Location: list.php');
    exit;
}

// Trouver les données du client et du mécanicien
$client = null;
$mechanic = null;

foreach($usersData['clients'] as $c) {
    if($c['id'] == $intervention['client_id']) {
        $client = $c;
        break;
    }
}

foreach($mechanicsData['mechanics'] as $m) {
    if($m['id'] == $intervention['mechanic_id']) {
        $mechanic = $m;
        break;
    }
}

// Timeline des étapes (simulé)
$timeline = [
    [
        'step' => 'requested',
        'title' => 'Demande créée',
        'description' => 'Intervention demandée par le client',
        'timestamp' => $intervention['created_at'],
        'completed' => true,
        'icon' => 'bi-plus-circle'
    ],
    [
        'step' => 'accepted',
        'title' => 'Acceptée par le mécanicien',
        'description' => 'Le mécanicien a accepté l\'intervention',
        'timestamp' => $intervention['accepted_at'] ?? null,
        'completed' => in_array($intervention['status'], ['accepted', 'in_progress', 'completed']),
        'icon' => 'bi-check-circle'
    ],
    [
        'step' => 'en_route',
        'title' => 'En route',
        'description' => 'Le mécanicien se dirige vers le lieu d\'intervention',
        'timestamp' => $intervention['started_at'] ?? null,
        'completed' => in_array($intervention['status'], ['in_progress', 'completed']),
        'icon' => 'bi-geo-alt'
    ],
    [
        'step' => 'arrived',
        'title' => 'Arrivé sur place',
        'description' => 'Le mécanicien est arrivé à destination',
        'timestamp' => $intervention['started_at'] ?? null,
        'completed' => in_array($intervention['status'], ['in_progress', 'completed']),
        'icon' => 'bi-pin-map'
    ],
    [
        'step' => 'in_progress',
        'title' => 'Intervention en cours',
        'description' => 'Travaux en cours',
        'timestamp' => $intervention['started_at'] ?? null,
        'completed' => $intervention['status'] === 'completed',
        'icon' => 'bi-tools'
    ],
    [
        'step' => 'completed',
        'title' => 'Terminée',
        'description' => 'Intervention finalisée avec succès',
        'timestamp' => $intervention['completed_at'] ?? null,
        'completed' => $intervention['status'] === 'completed',
        'icon' => 'bi-check-circle-fill'
    ]
];

// Messages du chat (simulé)
$messages = [
    [
        'sender' => 'system',
        'message' => 'Intervention créée automatiquement',
        'timestamp' => $intervention['created_at']
    ],
    [
        'sender' => 'mechanic',
        'message' => 'Je suis en route vers le lieu d\'intervention.',
        'timestamp' => $intervention['accepted_at'] ?? $intervention['created_at']
    ],
    [
        'sender' => 'mechanic',
        'message' => 'Je suis arrivé. Je commence le diagnostic.',
        'timestamp' => $intervention['started_at'] ?? $intervention['created_at']
    ]
];
?>

<!-- Page Header -->
<div class="page-header">
    <div class="page-header-content">
        <h1 class="page-title">
            <i class="bi bi-tools"></i>
            Intervention <?= e($intervention['id']) ?>
        </h1>
        <p class="page-subtitle">
            <span class="service-badge">
                <i class="bi bi-wrench"></i>
                <?= e($intervention['type']) ?>
            </span>
            •
            <span class="location-badge">
                <i class="bi bi-geo-alt"></i>
                <?= e($intervention['location']) ?>
            </span>
        </p>
    </div>
    <div class="page-header-actions">
        <div class="intervention-timer" id="interventionTimer">
            <?php if($intervention['status'] === 'in_progress'): ?>
            <div class="timer-display">
                <i class="bi bi-clock"></i>
                <span id="timerValue">00:00:00</span>
            </div>
            <?php endif; ?>
        </div>
        <?= getStatusBadge($intervention['status']) ?>
        <div class="btn-group">
            <button class="btn btn-outline-primary" onclick="updateStatus()">
                <i class="bi bi-arrow-repeat"></i>
                Changer statut
            </button>
            <button class="btn btn-outline-secondary" onclick="window.location.href='list.php'">
                <i class="bi bi-arrow-left"></i>
                Retour à la liste
            </button>
        </div>
    </div>
</div>

<div class="row g-4">
    <!-- Left Column -->
    <div class="col-xl-8">
        <!-- Client & Mechanic Cards -->
        <div class="row g-4 mb-4">
            <!-- Client Card -->
            <div class="col-md-6">
                <div class="dashboard-card">
                    <div class="card-header">
                        <h5 class="card-title">
                            <i class="bi bi-person"></i>
                            Client
                        </h5>
                    </div>
                    <div class="card-body">
                        <?php if($client): ?>
                        <div class="user-profile-card">
                            <div class="user-avatar-large">
                                <img src="<?= getDefaultAvatar($client['name']) ?>" alt="">
                            </div>
                            <div class="user-details">
                                <h6 class="user-name"><?= e($client['name']) ?></h6>
                                <div class="user-meta">
                                    <span class="meta-item">
                                        <i class="bi bi-telephone"></i>
                                        <?= e($client['phone']) ?>
                                    </span>
                                    <span class="meta-item">
                                        <i class="bi bi-envelope"></i>
                                        <?= e($client['email']) ?>
                                    </span>
                                    <span class="meta-item">
                                        <i class="bi bi-star-fill text-warning"></i>
                                        <?= $client['rating'] ?> (<?= $client['total_interventions'] ?> interventions)
                                    </span>
                                </div>
                            </div>
                            <div class="user-actions">
                                <button class="btn btn-sm btn-outline-primary" onclick="callClient()">
                                    <i class="bi bi-telephone"></i>
                                </button>
                                <button class="btn btn-sm btn-outline-primary" onclick="messageClient()">
                                    <i class="bi bi-chat"></i>
                                </button>
                                <button class="btn btn-sm btn-outline-secondary" onclick="viewClientProfile()">
                                    <i class="bi bi-eye"></i>
                                </button>
                            </div>
                        </div>
                        <?php else: ?>
                        <div class="text-center text-muted">
                            <i class="bi bi-person-x" style="font-size: 2rem;"></i>
                            <p>Informations client non disponibles</p>
                        </div>
                        <?php endif; ?>
                    </div>
                </div>
            </div>

            <!-- Mechanic Card -->
            <div class="col-md-6">
                <div class="dashboard-card">
                    <div class="card-header">
                        <h5 class="card-title">
                            <i class="bi bi-wrench"></i>
                            Mécanicien
                        </h5>
                    </div>
                    <div class="card-body">
                        <?php if($mechanic): ?>
                        <div class="user-profile-card">
                            <div class="user-avatar-large">
                                <img src="<?= getDefaultAvatar($mechanic['name']) ?>" alt="">
                                <div class="status-indicator status-<?= $mechanic['status'] ?>">
                                    <i class="bi bi-circle-fill"></i>
                                </div>
                            </div>
                            <div class="user-details">
                                <h6 class="user-name"><?= e($mechanic['name']) ?></h6>
                                <div class="user-meta">
                                    <span class="meta-item">
                                        <i class="bi bi-telephone"></i>
                                        <?= e($mechanic['phone']) ?>
                                    </span>
                                    <span class="meta-item">
                                        <i class="bi bi-geo-alt"></i>
                                        <?= e($mechanic['location']) ?>
                                    </span>
                                    <span class="meta-item">
                                        <i class="bi bi-star-fill text-warning"></i>
                                        <?= $mechanic['rating'] ?> (<?= $mechanic['total_reviews'] ?> avis)
                                    </span>
                                    <span class="meta-item">
                                        <i class="bi bi-tools"></i>
                                        <?= $mechanic['total_interventions'] ?> interventions
                                    </span>
                                </div>
                            </div>
                            <div class="user-actions">
                                <button class="btn btn-sm btn-outline-primary" onclick="callMechanic()">
                                    <i class="bi bi-telephone"></i>
                                </button>
                                <button class="btn btn-sm btn-outline-primary" onclick="messageMechanic()">
                                    <i class="bi bi-chat"></i>
                                </button>
                                <button class="btn btn-sm btn-outline-secondary" onclick="viewMechanicProfile()">
                                    <i class="bi bi-eye"></i>
                                </button>
                            </div>
                        </div>
                        <?php else: ?>
                        <div class="text-center text-muted">
                            <i class="bi bi-wrench" style="font-size: 2rem;"></i>
                            <p>Mécanicien non assigné</p>
                            <button class="btn btn-primary btn-sm" onclick="assignMechanic()">
                                Assigner un mécanicien
                            </button>
                        </div>
                        <?php endif; ?>
                    </div>
                </div>
            </div>
        </div>

        <!-- Timeline -->
        <div class="dashboard-card mb-4">
            <div class="card-header">
                <h5 class="card-title">
                    <i class="bi bi-timeline"></i>
                    Chronologie de l'intervention
                </h5>
            </div>
            <div class="card-body">
                <div class="intervention-timeline">
                    <?php foreach($timeline as $index => $step): ?>
                    <div class="timeline-item <?= $step['completed'] ? 'completed' : 'pending' ?>">
                        <div class="timeline-marker">
                            <div class="timeline-icon">
                                <i class="bi <?= $step['icon'] ?>"></i>
                            </div>
                            <?php if($index < count($timeline) - 1): ?>
                            <div class="timeline-line"></div>
                            <?php endif; ?>
                        </div>
                        <div class="timeline-content">
                            <div class="timeline-header">
                                <h6 class="timeline-title"><?= e($step['title']) ?></h6>
                                <?php if($step['timestamp']): ?>
                                <small class="timeline-time">
                                    <i class="bi bi-clock"></i>
                                    <?= timeAgo($step['timestamp']) ?>
                                </small>
                                <?php endif; ?>
                            </div>
                            <p class="timeline-description"><?= e($step['description']) ?></p>
                        </div>
                    </div>
                    <?php endforeach; ?>
                </div>
            </div>
        </div>

        <!-- Location Map -->
        <div class="dashboard-card">
            <div class="card-header">
                <h5 class="card-title">
                    <i class="bi bi-geo-alt"></i>
                    Localisation & Suivi
                </h5>
                <div class="card-header-actions">
                    <button class="btn btn-sm btn-outline-primary" onclick="refreshLocation()">
                        <i class="bi bi-arrow-clockwise"></i>
                        Actualiser
                    </button>
                </div>
            </div>
            <div class="card-body p-0">
                <div class="intervention-map" id="interventionMap" style="height: 300px;">
                    <!-- Google Maps sera intégré ici -->
                    <div class="map-placeholder">
                        <i class="bi bi-geo-alt" style="font-size: 3rem; color: var(--text-secondary);"></i>
                        <p class="text-muted mt-3">Carte de localisation de l'intervention</p>
                        <div class="location-info">
                            <strong><?= e($intervention['location']) ?></strong><br>
                            <small class="text-muted">Coordonnées: <?= $intervention['coordinates']['lat'] ?>, <?= $intervention['coordinates']['lng'] ?></small>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <!-- Right Column -->
    <div class="col-xl-4">
        <!-- Payment Section -->
        <div class="dashboard-card mb-4">
            <div class="card-header">
                <h5 class="card-title">
                    <i class="bi bi-currency-dollar"></i>
                    Paiement
                </h5>
            </div>
            <div class="card-body">
                <div class="payment-summary">
                    <div class="payment-row">
                        <span class="payment-label">Montant de l'intervention</span>
                        <span class="payment-value"><?= formatCurrency($intervention['amount']) ?></span>
                    </div>
                    <div class="payment-row">
                        <span class="payment-label">Commission (15%)</span>
                        <span class="payment-value">-<?= formatCurrency($intervention['commission']) ?></span>
                    </div>
                    <hr>
                    <div class="payment-row total">
                        <span class="payment-label">Revenus nets</span>
                        <span class="payment-value text-gold">
                            <?= formatCurrency($intervention['amount'] - $intervention['commission']) ?>
                        </span>
                    </div>
                </div>

                <div class="payment-status mt-3">
                    <div class="status-item">
                        <span class="status-label">Statut du paiement:</span>
                        <span class="status-badge status-<?= $intervention['payment_status'] ?>">
                            <?= ucfirst($intervention['payment_status']) ?>
                        </span>
                    </div>
                    <div class="status-item">
                        <span class="status-label">Méthode:</span>
                        <span class="status-value">
                            <?php
                            $methods = [
                                'wallet' => 'Portefeuille MECASOS',
                                'card' => 'Carte bancaire',
                                'cash' => 'Espèces'
                            ];
                            echo $methods[$intervention['payment_method']] ?? $intervention['payment_method'];
                            ?>
                        </span>
                    </div>
                </div>

                <?php if($intervention['payment_status'] === 'pending'): ?>
                <div class="payment-actions mt-3">
                    <button class="btn btn-success btn-sm w-100" onclick="processPayment()">
                        <i class="bi bi-check-circle"></i>
                        Marquer comme payé
                    </button>
                </div>
                <?php endif; ?>
            </div>
        </div>

        <!-- Chat/Notes Section -->
        <div class="dashboard-card mb-4">
            <div class="card-header">
                <h5 class="card-title">
                    <i class="bi bi-chat-dots"></i>
                    Communication Interne
                </h5>
            </div>
            <div class="card-body p-0">
                <div class="chat-container">
                    <div class="chat-messages" id="chatMessages">
                        <?php foreach($messages as $message): ?>
                        <div class="chat-message <?= $message['sender'] ?>">
                            <div class="message-content">
                                <p><?= e($message['message']) ?></p>
                                <small class="message-time">
                                    <i class="bi bi-clock"></i>
                                    <?= timeAgo($message['timestamp']) ?>
                                </small>
                            </div>
                        </div>
                        <?php endforeach; ?>
                    </div>
                    <div class="chat-input">
                        <form id="chatForm">
                            <div class="input-group">
                                <input type="text" class="form-control" placeholder="Tapez votre message..." id="chatInput">
                                <button class="btn btn-primary" type="submit">
                                    <i class="bi bi-send"></i>
                                </button>
                            </div>
                        </form>
                    </div>
                </div>
            </div>
        </div>

        <!-- Quick Actions -->
        <div class="dashboard-card">
            <div class="card-header">
                <h5 class="card-title">
                    <i class="bi bi-lightning"></i>
                    Actions Rapides
                </h5>
            </div>
            <div class="card-body">
                <div class="quick-actions-list">
                    <button class="quick-action-btn" onclick="updateStatus()">
                        <i class="bi bi-arrow-repeat"></i>
                        <span>Changer le statut</span>
                    </button>

                    <button class="quick-action-btn" onclick="assignMechanic()">
                        <i class="bi bi-person-plus"></i>
                        <span>Assigner mécanicien</span>
                    </button>

                    <button class="quick-action-btn" onclick="sendNotification()">
                        <i class="bi bi-bell"></i>
                        <span>Envoyer notification</span>
                    </button>

                    <button class="quick-action-btn" onclick="generateReport()">
                        <i class="bi bi-file-earmark-text"></i>
                        <span>Générer rapport</span>
                    </button>

                    <?php if(in_array($intervention['status'], ['pending', 'accepted'])): ?>
                    <button class="quick-action-btn text-danger" onclick="cancelIntervention()">
                        <i class="bi bi-x-circle"></i>
                        <span>Annuler l'intervention</span>
                    </button>
                    <?php endif; ?>
                </div>
            </div>
        </div>
    </div>
</div>

<script>
// ========================================
// INTERVENTION DETAILS FUNCTIONALITY
// ========================================
document.addEventListener('DOMContentLoaded', function() {
    // Initialize timer if intervention is in progress
    <?php if($intervention['status'] === 'in_progress' && $intervention['started_at']): ?>
    initializeTimer('<?= $intervention['started_at'] ?>');
    <?php endif; ?>

    // Chat form submission
    document.getElementById('chatForm').addEventListener('submit', function(e) {
        e.preventDefault();
        const input = document.getElementById('chatInput');
        const message = input.value.trim();

        if(message) {
            addChatMessage('admin', message);
            input.value = '';

            // Simulation de réponse automatique
            setTimeout(() => {
                addChatMessage('system', 'Message envoyé au mécanicien');
            }, 1000);
        }
    });
});

// Timer functionality
function initializeTimer(startTime) {
    const startTimestamp = new Date(startTime).getTime();

    function updateTimer() {
        const now = new Date().getTime();
        const elapsed = now - startTimestamp;

        const hours = Math.floor(elapsed / (1000 * 60 * 60));
        const minutes = Math.floor((elapsed % (1000 * 60 * 60)) / (1000 * 60));
        const seconds = Math.floor((elapsed % (1000 * 60)) / 1000);

        const timerValue = document.getElementById('timerValue');
        if(timerValue) {
            timerValue.textContent =
                String(hours).padStart(2, '0') + ':' +
                String(minutes).padStart(2, '0') + ':' +
                String(seconds).padStart(2, '0');
        }
    }

    updateTimer();
    setInterval(updateTimer, 1000);
}

// Chat functionality
function addChatMessage(sender, message) {
    const chatMessages = document.getElementById('chatMessages');
    const messageElement = document.createElement('div');
    messageElement.className = `chat-message ${sender}`;
    messageElement.innerHTML = `
        <div class="message-content">
            <p>${message}</p>
            <small class="message-time">
                <i class="bi bi-clock"></i>
                à l'instant
            </small>
        </div>
    `;

    chatMessages.appendChild(messageElement);
    chatMessages.scrollTop = chatMessages.scrollHeight;
}

// Action functions
function updateStatus() {
    // Simulation de changement de statut
    showNotification('Fonctionnalité de changement de statut en développement', 'info');
}

function assignMechanic() {
    showNotification('Ouverture du sélecteur de mécanicien...', 'info');
}

function callClient() {
    window.open(`tel:<?= $client['phone'] ?? '' ?>`);
}

function callMechanic() {
    window.open(`tel:<?= $mechanic['phone'] ?? '' ?>`);
}

function messageClient() {
    showNotification('Ouverture du chat client...', 'info');
}

function messageMechanic() {
    showNotification('Ouverture du chat mécanicien...', 'info');
}

function viewClientProfile() {
    window.location.href = `../users/user-details.php?id=<?= $client['id'] ?? '' ?>&type=client`;
}

function viewMechanicProfile() {
    window.location.href = `../users/user-details.php?id=<?= $mechanic['id'] ?? '' ?>&type=mechanic`;
}

function sendNotification() {
    showNotification('Ouverture de l\'éditeur de notifications...', 'info');
}

function generateReport() {
    showNotification('Génération du rapport en cours...', 'info');
    setTimeout(() => {
        showNotification('Rapport généré avec succès!', 'success');
    }, 2000);
}

function processPayment() {
    if(confirm('Marquer ce paiement comme effectué?')) {
        showNotification('Paiement marqué comme effectué!', 'success');
    }
}

function cancelIntervention() {
    if(confirm('Êtes-vous sûr de vouloir annuler cette intervention? Cette action est irréversible.')) {
        showNotification('Intervention annulée', 'warning');
    }
}

function refreshLocation() {
    showNotification('Actualisation de la localisation...', 'info');
    setTimeout(() => {
        showNotification('Localisation mise à jour!', 'success');
    }, 1500);
}

function showNotification(message, type = 'info') {
    const notification = document.createElement('div');
    notification.className = `alert alert-${type} alert-dismissible fade show position-fixed`;
    notification.style.cssText = 'top: 20px; right: 20px; z-index: 9999; min-width: 300px;';
    notification.innerHTML = `
        ${message}
        <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
    `;

    document.body.appendChild(notification);

    setTimeout(() => {
        if(notification.parentNode) {
            notification.remove();
        }
    }, 5000);
}
</script>

<style>
/* Intervention Details Styles */
.service-badge, .location-badge {
    display: inline-flex;
    align-items: center;
    gap: 6px;
    background: rgba(225, 160, 4, 0.1);
    color: var(--primary-color);
    padding: 4px 8px;
    border-radius: 4px;
    font-size: 12px;
    font-weight: 500;
}

.location-badge {
    background: rgba(59, 130, 246, 0.1);
    color: #3B82F6;
}

.intervention-timer {
    background: var(--card-background);
    border: 1px solid var(--border-color);
    border-radius: 8px;
    padding: 8px 16px;
    margin-right: 16px;
}

.timer-display {
    display: flex;
    align-items: center;
    gap: 8px;
    color: var(--primary-color);
    font-weight: 500;
}

#timerValue {
    font-family: 'Courier New', monospace;
    font-size: 16px;
}

/* User Profile Cards */
.user-profile-card {
    display: flex;
    flex-direction: column;
    align-items: center;
    text-align: center;
    gap: 16px;
}

.user-avatar-large {
    position: relative;
    width: 80px;
    height: 80px;
    border-radius: 50%;
    overflow: hidden;
    border: 3px solid var(--border-color);
}

.user-avatar-large img {
    width: 100%;
    height: 100%;
    object-fit: cover;
}

.status-indicator {
    position: absolute;
    bottom: 0;
    right: 0;
    width: 20px;
    height: 20px;
    border-radius: 50%;
    display: flex;
    align-items: center;
    justify-content: center;
    border: 2px solid var(--card-background);
}

.status-indicator.status-online i { color: #10B981; }
.status-indicator.status-busy i { color: #F59E0B; }
.status-indicator.status-offline i { color: #6B7280; }

.user-details {
    flex: 1;
}

.user-name {
    margin-bottom: 8px;
    color: var(--text-primary);
}

.user-meta {
    display: flex;
    flex-direction: column;
    gap: 4px;
}

.meta-item {
    display: flex;
    align-items: center;
    gap: 6px;
    font-size: 12px;
    color: var(--text-secondary);
}

.user-actions {
    display: flex;
    gap: 8px;
}

/* Timeline */
.intervention-timeline {
    position: relative;
    padding-left: 40px;
}

.timeline-item {
    position: relative;
    margin-bottom: 24px;
    opacity: 0.6;
}

.timeline-item.completed {
    opacity: 1;
}

.timeline-marker {
    position: absolute;
    left: -40px;
    top: 0;
    width: 32px;
    height: 32px;
    display: flex;
    align-items: center;
    justify-content: center;
}

.timeline-icon {
    width: 32px;
    height: 32px;
    border-radius: 50%;
    background: var(--border-color);
    display: flex;
    align-items: center;
    justify-content: center;
    color: var(--text-secondary);
    z-index: 2;
    position: relative;
}

.timeline-item.completed .timeline-icon {
    background: var(--primary-color);
    color: white;
}

.timeline-line {
    position: absolute;
    top: 32px;
    left: 50%;
    transform: translateX(-50%);
    width: 2px;
    height: calc(100% + 24px);
    background: var(--border-color);
}

.timeline-item.completed .timeline-line {
    background: var(--primary-color);
}

.timeline-content {
    background: rgba(255, 255, 255, 0.02);
    border-radius: 8px;
    padding: 16px;
}

.timeline-header {
    display: flex;
    justify-content: space-between;
    align-items: flex-start;
    margin-bottom: 8px;
}

.timeline-title {
    margin: 0;
    color: var(--text-primary);
}

.timeline-time {
    color: var(--text-secondary);
    font-size: 12px;
}

.timeline-description {
    margin: 0;
    color: var(--text-secondary);
    font-size: 14px;
}

/* Payment Section */
.payment-summary {
    display: flex;
    flex-direction: column;
    gap: 8px;
}

.payment-row {
    display: flex;
    justify-content: space-between;
    align-items: center;
}

.payment-row.total {
    font-weight: 600;
    font-size: 16px;
    padding-top: 8px;
}

.payment-label {
    color: var(--text-secondary);
}

.payment-value {
    color: var(--text-primary);
}

.payment-status {
    display: flex;
    flex-direction: column;
    gap: 8px;
}

.status-item {
    display: flex;
    justify-content: space-between;
    align-items: center;
}

.status-badge {
    padding: 2px 8px;
    border-radius: 4px;
    font-size: 11px;
    font-weight: 500;
}

.status-completed { background: #10B981; color: white; }
.status-pending { background: #F59E0B; color: white; }
.status-cancelled { background: #EF4444; color: white; }

/* Chat */
.chat-container {
    height: 300px;
    display: flex;
    flex-direction: column;
}

.chat-messages {
    flex: 1;
    overflow-y: auto;
    padding: 16px;
    display: flex;
    flex-direction: column;
    gap: 12px;
}

.chat-message {
    display: flex;
    margin-bottom: 8px;
}

.chat-message.system {
    justify-content: center;
}

.chat-message.mechanic {
    justify-content: flex-start;
}

.chat-message.admin {
    justify-content: flex-end;
}

.message-content {
    max-width: 70%;
    background: rgba(255, 255, 255, 0.05);
    border-radius: 8px;
    padding: 8px 12px;
}

.chat-message.admin .message-content {
    background: var(--primary-color);
}

.message-content p {
    margin: 0;
    color: var(--text-primary);
}

.chat-message.admin .message-content p {
    color: white;
}

.message-time {
    display: block;
    margin-top: 4px;
    font-size: 10px;
    color: var(--text-secondary);
}

.chat-message.admin .message-time {
    color: rgba(255, 255, 255, 0.7);
}

.chat-input {
    border-top: 1px solid var(--border-color);
    padding: 16px;
}

/* Quick Actions */
.quick-actions-list {
    display: flex;
    flex-direction: column;
    gap: 8px;
}

.quick-action-btn {
    display: flex;
    align-items: center;
    gap: 12px;
    padding: 12px 16px;
    background: rgba(255, 255, 255, 0.02);
    border: 1px solid var(--border-color);
    border-radius: 8px;
    color: var(--text-primary);
    text-decoration: none;
    transition: all 0.2s ease;
}

.quick-action-btn:hover {
    background: rgba(255, 255, 255, 0.05);
    border-color: var(--primary-color);
    color: var(--primary-color);
}

.quick-action-btn.text-danger:hover {
    background: rgba(239, 68, 68, 0.1);
    border-color: #EF4444;
    color: #EF4444;
}

/* Map Placeholder */
.intervention-map {
    position: relative;
    background: linear-gradient(135deg, #1d1d1d 0%, #2d2d2d 100%);
    border-radius: 8px;
    display: flex;
    align-items: center;
    justify-content: center;
}

.map-placeholder {
    text-align: center;
    color: var(--text-secondary);
}

.location-info {
    margin-top: 16px;
    padding: 12px;
    background: rgba(255, 255, 255, 0.05);
    border-radius: 6px;
}

/* Responsive */
@media (max-width: 768px) {
    .page-header-actions {
        flex-direction: column;
        gap: 12px;
    }

    .intervention-timer {
        margin-right: 0;
        margin-bottom: 8px;
    }

    .user-profile-card {
        flex-direction: row;
        text-align: left;
    }

    .user-avatar-large {
        flex-shrink: 0;
    }

    .user-actions {
        flex-direction: column;
        gap: 4px;
    }

    .timeline-header {
        flex-direction: column;
        gap: 4px;
    }

    .intervention-timeline {
        padding-left: 20px;
    }

    .timeline-marker {
        left: -20px;
    }

    .timeline-icon {
        width: 24px;
        height: 24px;
    }

    .timeline-icon i {
        font-size: 12px;
    }
}
</style>
