<?php
$pageTitle = 'Gestion des Notifications';
$breadcrumbs = [
    ['label' => 'Dashboard', 'url' => '../index.php'],
    ['label' => 'Notifications', 'url' => '#']
];

require_once __DIR__ . '/../../config/constants.php';
require_once __DIR__ . '/../../config/functions.php';

// Charger les données mock
$usersData = json_decode(file_get_contents(__DIR__ . '/../../mockdata/users.json'), true);
$mechanicsData = json_decode(file_get_contents(__DIR__ . '/../../mockdata/mechanics.json'), true);

// Notifications mock
$notifications = [
    [
        'id' => 'NOT-001',
        'title' => 'Maintenance programmée',
        'message' => 'L\'application sera en maintenance ce dimanche de 2h à 4h.',
        'type' => 'maintenance',
        'recipients' => 'all_users',
        'recipient_count' => 1247,
        'sent_at' => '2025-11-20 10:00:00',
        'open_rate' => 78.5,
        'status' => 'sent'
    ],
    [
        'id' => 'NOT-002',
        'title' => 'Nouvelle fonctionnalité disponible',
        'message' => 'Découvrez le suivi GPS en temps réel pour vos interventions!',
        'type' => 'feature',
        'recipients' => 'all_users',
        'recipient_count' => 1247,
        'sent_at' => '2025-11-18 14:30:00',
        'open_rate' => 65.2,
        'status' => 'sent'
    ],
    [
        'id' => 'NOT-003',
        'title' => 'Promotion spéciale mécaniciens',
        'message' => 'Commission augmentée de 2% ce mois-ci pour tous les mécaniciens actifs.',
        'type' => 'promotion',
        'recipients' => 'mechanics',
        'recipient_count' => 342,
        'sent_at' => '2025-11-15 09:00:00',
        'open_rate' => 89.3,
        'status' => 'sent'
    ],
    [
        'id' => 'NOT-004',
        'title' => 'Mise à jour des conditions générales',
        'message' => 'Consultez les nouvelles conditions générales d\'utilisation.',
        'type' => 'legal',
        'recipients' => 'all_users',
        'recipient_count' => 1247,
        'sent_at' => '2025-11-10 16:00:00',
        'open_rate' => 45.8,
        'status' => 'sent'
    ],
    [
        'id' => 'NOT-005',
        'title' => 'Campagne de fidélisation',
        'message' => 'Bénéficiez de 10% de réduction sur votre prochaine intervention!',
        'type' => 'promotion',
        'recipients' => 'clients',
        'recipient_count' => 905,
        'scheduled_at' => '2025-11-25 08:00:00',
        'status' => 'scheduled'
    ]
];

// Templates de notifications
$templates = [
    [
        'id' => 'template_maintenance',
        'name' => 'Maintenance',
        'title' => 'Maintenance programmée',
        'message' => 'L\'application sera en maintenance le [DATE] de [HEURE_DÉBUT] à [HEURE_FIN].',
        'type' => 'maintenance'
    ],
    [
        'id' => 'template_promotion',
        'name' => 'Promotion',
        'title' => 'Offre spéciale',
        'message' => 'Profitez de [POURCENTAGE]% de réduction sur vos interventions jusqu\'au [DATE_FIN]!',
        'type' => 'promotion'
    ],
    [
        'id' => 'template_welcome',
        'name' => 'Bienvenue',
        'title' => 'Bienvenue sur MECASOS',
        'message' => 'Bienvenue [NOM]! Votre compte a été créé avec succès.',
        'type' => 'welcome'
    ],
    [
        'id' => 'template_intervention',
        'name' => 'Intervention',
        'title' => 'Intervention confirmée',
        'message' => 'Votre intervention avec [MÉCANICIEN] est confirmée pour le [DATE] à [HEURE].',
        'type' => 'intervention'
    ]
];

// Statistiques
$stats = [
    'total_sent' => count(array_filter($notifications, fn($n) => $n['status'] === 'sent')),
    'total_scheduled' => count(array_filter($notifications, fn($n) => $n['status'] === 'scheduled')),
    'avg_open_rate' => 69.7,
    'total_recipients' => array_sum(array_column($notifications, 'recipient_count'))
];
?>

<!-- Page Header -->
<div class="page-header">
    <div class="page-header-content">
        <h1 class="page-title">
            <i class="bi bi-bell"></i>
            Notifications
        </h1>
        <p class="page-subtitle">Envoyez et gérez les notifications push et emails</p>
    </div>
    <div class="page-header-actions">
        <div class="notification-stats">
            <div class="stat-item">
                <span class="stat-value"><?= $stats['total_sent'] ?></span>
                <span class="stat-label">Envoyées</span>
            </div>
            <div class="stat-item">
                <span class="stat-value text-warning"><?= $stats['total_scheduled'] ?></span>
                <span class="stat-label">Programmées</span>
            </div>
            <div class="stat-item">
                <span class="stat-value text-gold"><?= number_format($stats['avg_open_rate'], 1) ?>%</span>
                <span class="stat-label">Taux d'ouverture</span>
            </div>
        </div>
        <button class="btn btn-primary btn-icon-text" data-bs-toggle="modal" data-bs-target="#composeModal">
            <i class="bi bi-plus-lg"></i>
            <span>Nouvelle notification</span>
        </button>
    </div>
</div>

<div class="row g-4">
    <!-- Compose Section -->
    <div class="col-xl-4">
        <div class="dashboard-card">
            <div class="card-header">
                <h5 class="card-title">
                    <i class="bi bi-pencil-square"></i>
                    Composer
                </h5>
            </div>
            <div class="card-body">
                <form id="quickComposeForm">
                    <div class="mb-3">
                        <label class="form-label">Destinataires</label>
                        <select class="form-select" name="recipients" required>
                            <option value="">Sélectionner...</option>
                            <option value="all_users">Tous les utilisateurs (<?= $usersData['stats']['total'] + $mechanicsData['stats']['total'] ?>)</option>
                            <option value="clients">Clients uniquement (<?= $usersData['stats']['total'] ?>)</option>
                            <option value="mechanics">Mécaniciens uniquement (<?= $mechanicsData['stats']['total'] ?>)</option>
                            <option value="active_mechanics">Mécaniciens actifs (<?= $mechanicsData['stats']['online'] ?>)</option>
                        </select>
                    </div>

                    <div class="mb-3">
                        <label class="form-label">Type</label>
                        <select class="form-select" name="type" required>
                            <option value="general">Général</option>
                            <option value="urgent">Urgent</option>
                            <option value="promotion">Promotion</option>
                            <option value="maintenance">Maintenance</option>
                        </select>
                    </div>

                    <div class="mb-3">
                        <label class="form-label">Titre</label>
                        <input type="text" class="form-control" name="title" placeholder="Titre de la notification" required>
                    </div>

                    <div class="mb-3">
                        <label class="form-label">Message</label>
                        <textarea class="form-control" name="message" rows="4" placeholder="Contenu de la notification..." required></textarea>
                    </div>

                    <div class="mb-3">
                        <label class="form-label">Programmer l'envoi</label>
                        <input type="datetime-local" class="form-control" name="scheduled_at">
                        <small class="text-muted">Laisser vide pour envoyer immédiatement</small>
                    </div>

                    <div class="d-grid">
                        <button type="submit" class="btn btn-primary">
                            <i class="bi bi-send"></i>
                            Envoyer
                        </button>
                    </div>
                </form>
            </div>
        </div>

        <!-- Templates -->
        <div class="dashboard-card mt-4">
            <div class="card-header">
                <h5 class="card-title">
                    <i class="bi bi-file-earmark-text"></i>
                    Templates
                </h5>
            </div>
            <div class="card-body">
                <div class="templates-list">
                    <?php foreach($templates as $template): ?>
                    <div class="template-item" onclick="loadTemplate('<?= $template['id'] ?>')">
                        <div class="template-icon">
                            <i class="bi bi-<?= getTemplateIcon($template['type']) ?>"></i>
                        </div>
                        <div class="template-info">
                            <span class="template-name"><?= e($template['name']) ?></span>
                            <small class="template-type"><?= ucfirst($template['type']) ?></small>
                        </div>
                        <div class="template-arrow">
                            <i class="bi bi-chevron-right"></i>
                        </div>
                    </div>
                    <?php endforeach; ?>
                </div>
            </div>
        </div>
    </div>

    <!-- History Table -->
    <div class="col-xl-8">
        <div class="dashboard-card">
            <div class="card-header">
                <div class="card-header-content">
                    <h5 class="card-title">
                        <i class="bi bi-clock-history"></i>
                        Historique
                    </h5>
                    <p class="card-subtitle">
                        <?= count($notifications) ?> notifications • <?= $stats['total_recipients'] ?> destinataires
                    </p>
                </div>
                <div class="card-header-actions">
                    <div class="btn-group" role="group">
                        <button type="button" class="btn btn-sm btn-outline-secondary active" data-filter="all">Toutes</button>
                        <button type="button" class="btn btn-sm btn-outline-secondary" data-filter="sent">Envoyées</button>
                        <button type="button" class="btn btn-sm btn-outline-secondary" data-filter="scheduled">Programmées</button>
                    </div>
                </div>
            </div>
            <div class="card-body p-0">
                <div class="table-responsive">
                    <table class="data-table">
                        <thead>
                            <tr>
                                <th>Titre</th>
                                <th>Destinataires</th>
                                <th>Envoyé</th>
                                <th>Taux d'ouverture</th>
                                <th>Actions</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach($notifications as $notification): ?>
                            <tr class="notification-row" data-status="<?= $notification['status'] ?>">
                                <td>
                                    <div class="notification-info">
                                        <div class="notification-title">
                                            <span class="badge badge-<?= $notification['type'] ?> me-2">
                                                <i class="bi bi-<?= getNotificationIcon($notification['type']) ?>"></i>
                                            </span>
                                            <?= e($notification['title']) ?>
                                        </div>
                                        <small class="notification-preview text-muted">
                                            <?= substr(e($notification['message']), 0, 50) ?>...
                                        </small>
                                    </div>
                                </td>
                                <td>
                                    <div class="recipients-info">
                                        <span class="recipient-type">
                                            <?= getRecipientLabel($notification['recipients']) ?>
                                        </span>
                                        <small class="recipient-count text-muted">
                                            <?= $notification['recipient_count'] ?> utilisateurs
                                        </small>
                                    </div>
                                </td>
                                <td>
                                    <?php if(isset($notification['sent_at'])): ?>
                                        <small class="text-muted">
                                            <i class="bi bi-send"></i>
                                            <?= timeAgo($notification['sent_at']) ?>
                                        </small>
                                    <?php elseif(isset($notification['scheduled_at'])): ?>
                                        <small class="text-warning">
                                            <i class="bi bi-clock"></i>
                                            Programmée pour<br>
                                            <?= formatDate($notification['scheduled_at'], 'd/m/Y H:i') ?>
                                        </small>
                                    <?php endif; ?>
                                </td>
                                <td>
                                    <?php if($notification['status'] === 'sent'): ?>
                                        <div class="open-rate">
                                            <span class="rate-value <?= $notification['open_rate'] >= 70 ? 'text-success' : ($notification['open_rate'] >= 50 ? 'text-warning' : 'text-danger') ?>">
                                                <?= number_format($notification['open_rate'], 1) ?>%
                                            </span>
                                            <div class="rate-bar">
                                                <div class="rate-fill" style="width: <?= $notification['open_rate'] ?>%"></div>
                                            </div>
                                        </div>
                                    <?php else: ?>
                                        <span class="text-muted">-</span>
                                    <?php endif; ?>
                                </td>
                                <td>
                                    <div class="btn-group" role="group">
                                        <button class="btn btn-sm btn-icon btn-outline-primary" title="Voir détails"
                                                onclick="viewNotification('<?= $notification['id'] ?>')">
                                            <i class="bi bi-eye"></i>
                                        </button>
                                        <?php if($notification['status'] === 'scheduled'): ?>
                                        <button class="btn btn-sm btn-icon btn-outline-warning" title="Modifier"
                                                onclick="editNotification('<?= $notification['id'] ?>')">
                                            <i class="bi bi-pencil"></i>
                                        </button>
                                        <button class="btn btn-sm btn-icon btn-outline-danger" title="Annuler"
                                                onclick="cancelNotification('<?= $notification['id'] ?>')">
                                            <i class="bi bi-x"></i>
                                        </button>
                                        <?php elseif($notification['status'] === 'sent'): ?>
                                        <button class="btn btn-sm btn-icon btn-outline-secondary" title="Dupliquer"
                                                onclick="duplicateNotification('<?= $notification['id'] ?>')">
                                            <i class="bi bi-copy"></i>
                                        </button>
                                        <?php endif; ?>
                                    </div>
                                </td>
                            </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>
            </div>
        </div>
    </div>
</div>

<!-- Modal: Compose Notification -->
<div class="modal fade" id="composeModal" tabindex="-1">
    <div class="modal-dialog modal-xl">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title">
                    <i class="bi bi-pencil-square"></i>
                    Nouvelle Notification
                </h5>
                <button type="button" class="btn-close btn-close-white" data-bs-dismiss="modal"></button>
            </div>
            <form id="composeForm">
                <div class="modal-body">
                    <div class="row g-3">
                        <div class="col-md-6">
                            <label class="form-label">Destinataires</label>
                            <select class="form-select" name="recipients" required>
                                <option value="">Sélectionner...</option>
                                <option value="all_users">Tous les utilisateurs</option>
                                <option value="clients">Clients uniquement</option>
                                <option value="mechanics">Mécaniciens uniquement</option>
                                <option value="active_mechanics">Mécaniciens actifs</option>
                                <option value="specific">Utilisateurs spécifiques</option>
                            </select>
                        </div>

                        <div class="col-md-6">
                            <label class="form-label">Type de notification</label>
                            <select class="form-select" name="type" required>
                                <option value="general">Général</option>
                                <option value="urgent">Urgent</option>
                                <option value="promotion">Promotion</option>
                                <option value="maintenance">Maintenance</option>
                                <option value="welcome">Bienvenue</option>
                            </select>
                        </div>

                        <div class="col-12">
                            <label class="form-label">Titre</label>
                            <input type="text" class="form-control" name="title" placeholder="Titre accrocheur" required>
                        </div>

                        <div class="col-12">
                            <label class="form-label">Message</label>
                            <textarea class="form-control" name="message" rows="4" placeholder="Contenu de la notification..." required></textarea>
                            <small class="text-muted">Maximum 160 caractères recommandés pour les notifications push</small>
                        </div>

                        <div class="col-md-6">
                            <label class="form-label">Image (optionnel)</label>
                            <input type="file" class="form-control" name="image" accept="image/*">
                            <small class="text-muted">Formats acceptés: JPG, PNG, GIF (Max 2MB)</small>
                        </div>

                        <div class="col-md-6">
                            <label class="form-label">Programmation</label>
                            <input type="datetime-local" class="form-control" name="scheduled_at">
                            <small class="text-muted">Laisser vide pour envoi immédiat</small>
                        </div>
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Annuler</button>
                    <button type="submit" class="btn btn-primary">
                        <i class="bi bi-send"></i>
                        Envoyer la notification
                    </button>
                </div>
            </form>
        </div>
    </div>
</div>

<?php
function getTemplateIcon($type) {
    $icons = [
        'maintenance' => 'tools',
        'promotion' => 'tag',
        'welcome' => 'person-plus',
        'intervention' => 'wrench'
    ];
    return $icons[$type] ?? 'bell';
}

function getNotificationIcon($type) {
    $icons = [
        'maintenance' => 'tools',
        'feature' => 'star',
        'promotion' => 'tag',
        'legal' => 'file-text'
    ];
    return $icons[$type] ?? 'bell';
}

function getRecipientLabel($recipients) {
    $labels = [
        'all_users' => 'Tous',
        'clients' => 'Clients',
        'mechanics' => 'Mécaniciens'
    ];
    return $labels[$recipients] ?? $recipients;
}
?>

<script>
// ========================================
// NOTIFICATIONS FUNCTIONALITY
// ========================================
document.addEventListener('DOMContentLoaded', function() {
    // Filter buttons
    document.querySelectorAll('[data-filter]').forEach(button => {
        button.addEventListener('click', function() {
            // Remove active class from all buttons
            document.querySelectorAll('[data-filter]').forEach(btn => btn.classList.remove('active'));
            this.classList.add('active');

            const filter = this.getAttribute('data-filter');
            filterNotifications(filter);
        });
    });

    // Quick compose form
    document.getElementById('quickComposeForm').addEventListener('submit', function(e) {
        e.preventDefault();
        submitNotification(this, false);
    });

    // Full compose form
    document.getElementById('composeForm').addEventListener('submit', function(e) {
        e.preventDefault();
        submitNotification(this, true);
    });
});

function filterNotifications(filter) {
    const rows = document.querySelectorAll('.notification-row');

    rows.forEach(row => {
        if (filter === 'all') {
            row.style.display = '';
        } else {
            row.style.display = row.getAttribute('data-status') === filter ? '' : 'none';
        }
    });
}

function loadTemplate(templateId) {
    // Simulation du chargement de template
    const templates = <?= json_encode($templates) ?>;
    const template = templates.find(t => t.id === templateId);

    if (template) {
        // Ouvrir le modal de composition et pré-remplir
        const modal = new bootstrap.Modal(document.getElementById('composeModal'));
        modal.show();

        // Pré-remplir le formulaire
        setTimeout(() => {
            document.querySelector('#composeForm [name="title"]').value = template.title;
            document.querySelector('#composeForm [name="message"]').value = template.message;
            document.querySelector('#composeForm [name="type"]').value = template.type;
        }, 100);

        showNotification('Template chargé!', 'success');
    }
}

function submitNotification(form, isModal = false) {
    const formData = new FormData(form);
    const data = Object.fromEntries(formData);

    // Validation basique
    if (!data.title || !data.message || !data.recipients) {
        showNotification('Veuillez remplir tous les champs requis', 'danger');
        return;
    }

    // Simulation d'envoi
    showNotification('Notification en cours d\'envoi...', 'info');

    setTimeout(() => {
        if (isModal) {
            const modal = bootstrap.Modal.getInstance(document.getElementById('composeModal'));
            modal.hide();
            form.reset();
        } else {
            form.reset();
        }

        showNotification('Notification envoyée avec succès!', 'success');

        // Recharger la page après un court délai pour simuler l'ajout
        setTimeout(() => {
            location.reload();
        }, 1000);
    }, 2000);
}

function viewNotification(id) {
    showNotification('Ouverture des détails de la notification...', 'info');
}

function editNotification(id) {
    showNotification('Ouverture de l\'éditeur...', 'info');
}

function cancelNotification(id) {
    if (confirm('Êtes-vous sûr de vouloir annuler cette notification programmée?')) {
        showNotification('Notification annulée', 'warning');
    }
}

function duplicateNotification(id) {
    showNotification('Duplication de la notification...', 'info');
    setTimeout(() => {
        showNotification('Notification dupliquée!', 'success');
    }, 1000);
}

function showNotification(message, type = 'info') {
    const notification = document.createElement('div');
    notification.className = `alert alert-${type} alert-dismissible fade show position-fixed`;
    notification.style.cssText = 'top: 20px; right: 20px; z-index: 9999; min-width: 300px;';
    notification.innerHTML = `
        ${message}
        <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
    `;

    document.body.appendChild(notification);

    setTimeout(() => {
        if(notification.parentNode) {
            notification.remove();
        }
    }, 5000);
}
</script>

<style>
/* Notification Styles */
.notification-stats {
    display: flex;
    gap: 24px;
    margin-right: 24px;
}

.stat-item {
    text-align: center;
}

.stat-value {
    display: block;
    font-size: 20px;
    font-weight: 700;
    color: var(--text-primary);
}

.stat-label {
    font-size: 12px;
    color: var(--text-secondary);
    text-transform: uppercase;
    letter-spacing: 0.5px;
}

/* Templates */
.templates-list {
    display: flex;
    flex-direction: column;
    gap: 8px;
}

.template-item {
    display: flex;
    align-items: center;
    gap: 12px;
    padding: 12px;
    border-radius: 8px;
    background: rgba(255, 255, 255, 0.02);
    cursor: pointer;
    transition: background 0.2s ease;
}

.template-item:hover {
    background: rgba(255, 255, 255, 0.05);
}

.template-icon {
    width: 40px;
    height: 40px;
    border-radius: 8px;
    background: var(--primary-color);
    display: flex;
    align-items: center;
    justify-content: center;
    color: white;
}

.template-info {
    flex: 1;
}

.template-name {
    display: block;
    font-weight: 500;
    color: var(--text-primary);
}

.template-type {
    color: var(--text-secondary);
    font-size: 12px;
}

.template-arrow {
    color: var(--text-secondary);
}

/* Data Table */
.data-table {
    width: 100%;
    margin-bottom: 0;
}

.data-table thead th {
    background: var(--card-background);
    border-bottom: 2px solid var(--border-color);
    font-weight: 600;
    color: var(--text-primary);
    padding: 16px 12px;
    text-align: left;
}

.data-table tbody td {
    padding: 16px 12px;
    border-bottom: 1px solid rgba(255, 255, 255, 0.05);
    vertical-align: middle;
}

.data-table tbody tr:hover {
    background: rgba(255, 255, 255, 0.02);
}

.notification-info {
    max-width: 300px;
}

.notification-title {
    font-weight: 500;
    color: var(--text-primary);
    margin-bottom: 4px;
}

.notification-preview {
    display: block;
    font-size: 12px;
}

.badge {
    font-size: 10px;
    padding: 2px 6px;
}

.badge-general { background: #6B7280; }
.badge-urgent { background: #EF4444; }
.badge-promotion { background: #10B981; }
.badge-maintenance { background: #F59E0B; }
.badge-legal { background: #3B82F6; }

.recipients-info {
    display: flex;
    flex-direction: column;
}

.recipient-type {
    font-weight: 500;
    color: var(--text-primary);
}

.recipient-count {
    font-size: 12px;
}

.open-rate {
    display: flex;
    flex-direction: column;
    align-items: flex-start;
    gap: 4px;
}

.rate-value {
    font-weight: 600;
    font-size: 14px;
}

.rate-bar {
    width: 60px;
    height: 4px;
    background: rgba(255, 255, 255, 0.1);
    border-radius: 2px;
    overflow: hidden;
}

.rate-fill {
    height: 100%;
    background: var(--primary-color);
    border-radius: 2px;
}

.btn-group .btn {
    padding: 6px 8px;
}

.btn-icon {
    width: 32px;
    height: 32px;
    display: flex;
    align-items: center;
    justify-content: center;
    padding: 0;
}

/* Modal Styles */
.modal-content {
    background: var(--card-background);
    border: 1px solid var(--border-color);
}

.modal-header {
    border-bottom: 1px solid var(--border-color);
}

.modal-footer {
    border-top: 1px solid var(--border-color);
}

/* Responsive */
@media (max-width: 768px) {
    .page-header-actions {
        flex-direction: column;
        gap: 12px;
    }

    .notification-stats {
        margin-right: 0;
        margin-bottom: 12px;
    }

    .templates-list {
        gap: 4px;
    }

    .template-item {
        padding: 8px;
    }

    .data-table {
        font-size: 14px;
    }

    .data-table thead th,
    .data-table tbody td {
        padding: 8px 6px;
    }

    .notification-info {
        max-width: none;
    }

    .btn-group {
        flex-direction: column;
        gap: 4px;
    }

    .btn-group .btn {
        width: 100%;
    }
}
</style>
