/**
 * MECASOS Admin Dashboard - Mechanics Management
 * Gestion avancée des mécaniciens professionnels
 */

let mechanicsData = [];
let selectedMechanics = [];
let dataTable;

document.addEventListener('DOMContentLoaded', function() {
    
    // ========================================
    // INITIALIZATION
    // ========================================
    
    initDataTable();
    initFilters();
    initCheckboxes();
    initFormValidation();
    initViewToggle();
    
    // ========================================
    // DATATABLES INITIALIZATION
    // ========================================
    
    function initDataTable() {
        dataTable = $('#mechanicsTable').DataTable({
            responsive: true,
            processing: true,
            language: {
                url: '//cdn.datatables.net/plug-ins/1.13.7/i18n/fr-FR.json',
                processing: '<div class="spinner-border text-primary" role="status"><span class="visually-hidden">Chargement...</span></div>'
            },
            pageLength: 15,
            lengthMenu: [[10, 15, 25, 50, -1], [10, 15, 25, 50, "Tous"]],
            order: [[13, 'desc']], // Trier par dernière activité
            columnDefs: [
                { 
                    orderable: false, 
                    targets: [0, 14] // Checkbox et Actions
                },
                {
                    targets: [10], // Revenus
                    render: function(data, type, row) {
                        if (type === 'sort' || type === 'type') {
                            return parseFloat(data.replace(/[^\d]/g, ''));
                        }
                        return data;
                    }
                }
            ],
            dom: '<"row"<"col-sm-12 col-md-6"l><"col-sm-12 col-md-6"f>>rt<"row"<"col-sm-12 col-md-5"i><"col-sm-12 col-md-7"p>>',
            buttons: [
                {
                    extend: 'excelHtml5',
                    text: '<i class="bi bi-file-excel"></i> Excel',
                    className: 'btn btn-sm btn-success',
                    title: 'Mécaniciens MECASOS - ' + new Date().toLocaleDateString('fr-FR'),
                    exportOptions: {
                        columns: [1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13]
                    }
                },
                {
                    extend: 'pdfHtml5',
                    text: '<i class="bi bi-file-pdf"></i> PDF',
                    className: 'btn btn-sm btn-danger',
                    title: 'Mécaniciens MECASOS',
                    orientation: 'landscape',
                    exportOptions: {
                        columns: [1, 2, 3, 4, 5, 8, 10]
                    }
                },
                {
                    extend: 'csvHtml5',
                    text: '<i class="bi bi-file-csv"></i> CSV',
                    className: 'btn btn-sm btn-info',
                    exportOptions: {
                        columns: [1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13]
                    }
                }
            ],
            drawCallback: function() {
                updateCheckboxStates();
            },
            initComplete: function() {
                console.log('Mechanics DataTable initialisée');
                animateTableRows();
            }
        });
        
        $('.dt-buttons').hide();
    }
    
    // ========================================
    // FILTERS MANAGEMENT
    // ========================================
    
    function initFilters() {
        // Recherche globale
        $('#globalSearch').on('keyup', debounce(function() {
            dataTable.search(this.value).draw();
        }, 300));
        
        // Filtre par statut en ligne
        $('#onlineStatusFilter').on('change', function() {
            const status = this.value;
            dataTable.column(4).search(status).draw();
        });
        
        // Filtre par vérification
        $('#verificationFilter').on('change', function() {
            const verification = this.value;
            dataTable.column(5).search(verification).draw();
        });
        
        // Filtre par spécialité (recherche dans la colonne)
        $('#specialtyFilter').on('change', function() {
            const specialty = this.value;
            dataTable.column(6).search(specialty).draw();
        });
        
        // Filtre par zone
        $('#zoneFilter').on('change', function() {
            const zone = this.value;
            dataTable.column(2).search(zone).draw();
        });
        
        // Réinitialiser les filtres
        $('#resetFilters').on('click', function() {
            $('#globalSearch').val('');
            $('#onlineStatusFilter').val('');
            $('#verificationFilter').val('');
            $('#specialtyFilter').val('');
            $('#zoneFilter').val('');
            dataTable.search('').columns().search('').draw();
            showToast('Filtres réinitialisés', 'info');
        });
        
        // Actualiser le tableau
        $('#refreshTable').on('click', function() {
            const btn = $(this);
            const icon = btn.find('i');
            
            icon.addClass('rotate-animation');
            setTimeout(() => {
                icon.removeClass('rotate-animation');
                dataTable.draw();
                showToast('Tableau actualisé', 'success');
            }, 1000);
        });
    }
    
    // ========================================
    // VIEW TOGGLE (Table / Grid)
    // ========================================
    
    function initViewToggle() {
        $('.view-toggle button').on('click', function() {
            const view = $(this).data('view');
            $('.view-toggle button').removeClass('active');
            $(this).addClass('active');
            
            if (view === 'grid') {
                showToast('Vue grille en cours de développement', 'info');
                // TODO: Implémenter vue grille
            }
        });
    }
    
    // ========================================
    // CHECKBOXES MANAGEMENT
    // ========================================
    
    function initCheckboxes() {
        $('#selectAll').on('change', function() {
            const isChecked = this.checked;
            $('.row-checkbox:visible').prop('checked', isChecked);
            updateSelectedMechanics();
        });
        
        $(document).on('change', '.row-checkbox', function() {
            updateSelectedMechanics();
            
            const totalVisible = $('.row-checkbox:visible').length;
            const totalChecked = $('.row-checkbox:checked').length;
            $('#selectAll').prop('checked', totalVisible === totalChecked && totalVisible > 0);
        });
    }
    
    function updateSelectedMechanics() {
        selectedMechanics = [];
        $('.row-checkbox:checked').each(function() {
            selectedMechanics.push(parseInt($(this).val()));
        });
        
        if (selectedMechanics.length > 0) {
            $('#bulkActionsBar').slideDown(300);
            $('#selectedCount').text(selectedMechanics.length);
        } else {
            $('#bulkActionsBar').slideUp(300);
        }
    }
    
    function updateCheckboxStates() {
        $('.row-checkbox').each(function() {
            const mechanicId = parseInt($(this).val());
            $(this).prop('checked', selectedMechanics.includes(mechanicId));
        });
    }
    
    // ========================================
    // FORM VALIDATION
    // ========================================
    
    function initFormValidation() {
        const form = document.getElementById('addMechanicForm');
        
        if (form) {
            form.addEventListener('submit', function(e) {
                e.preventDefault();
                
                const formData = new FormData(form);
                
                // Validation email
                const email = formData.get('email');
                if (!isValidEmail(email)) {
                    showToast('Email invalide', 'error');
                    return false;
                }
                
                // Validation téléphone
                const phone = formData.get('phone');
                if (!isValidPhone(phone)) {
                    showToast('Numéro de téléphone invalide', 'error');
                    return false;
                }
                
                // Vérifier qu'au moins une spécialité est sélectionnée
                const specialties = formData.getAll('specialties[]');
                if (specialties.length === 0) {
                    showToast('Veuillez sélectionner au moins une spécialité', 'error');
                    return false;
                }
                
                submitAddMechanicForm(formData);
            });
        }
    }
    
    function submitAddMechanicForm(formData) {
        toggleLoading(true);
        
        // Simulation d'appel API
        setTimeout(() => {
            toggleLoading(false);
            
            const modal = bootstrap.Modal.getInstance(document.getElementById('addMechanicModal'));
            modal.hide();
            
            document.getElementById('addMechanicForm').reset();
            dataTable.draw();
            
            showToast('Mécanicien créé avec succès ! En attente de vérification.', 'success');
        }, 1500);
    }
    
    // ========================================
    // UTILITIES
    // ========================================
    
    function animateTableRows() {
        $('.mechanics-table tbody tr').each(function(index) {
            $(this).css({
                opacity: 0,
                transform: 'translateY(20px)'
            }).delay(index * 30).animate({
                opacity: 1
            }, 300, function() {
                $(this).css('transform', 'translateY(0)');
            });
        });
    }
    
    function debounce(func, wait) {
        let timeout;
        return function executedFunction(...args) {
            const later = () => {
                clearTimeout(timeout);
                func(...args);
            };
            clearTimeout(timeout);
            timeout = setTimeout(later, wait);
        };
    }
    
    function isValidEmail(email) {
        const re = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
        return re.test(email);
    }
    
    function isValidPhone(phone) {
        const re = /^\+?237\s?[67]\d{8}$/;
        return re.test(phone.replace(/\s/g, ''));
    }
    
});

// ============================================
// MECHANIC ACTIONS (Global Functions)
// ============================================

/**
 * Voir les détails d'un mécanicien
 */
function viewMechanic(mechanicId) {
    toggleLoading(true);
    
    setTimeout(() => {
        const modalContent = generateMechanicDetailsHTML(mechanicId);
        document.getElementById('mechanicDetailsContent').innerHTML = modalContent;
        
        const modal = new bootstrap.Modal(document.getElementById('viewMechanicModal'));
        modal.show();
        
        toggleLoading(false);
    }, 800);
}

function generateMechanicDetailsHTML(mechanicId) {
    // Mock data - à remplacer par appel API
    const mechanic = {
        id: mechanicId,
        name: "Pierre Nkolo",
        email: "pierre.nkolo@mecasos.com",
        phone: "+237 690 234 567",
        status: "online",
        verification_status: "verified",
        location: "Douala, Akwa",
        coverage_zone: "Douala - Rayon 10km",
        registration_date: "2024-02-10 09:00:00",
        last_active: "2025-11-21 07:30:00",
        total_interventions: 127,
        completed_interventions: 121,
        cancelled_interventions: 6,
        rating: 4.9,
        total_reviews: 98,
        total_earned: 1850000,
        pending_commission: 275000,
        available_balance: 1575000,
        response_time_avg: "12 min",
        completion_rate: 95.3,
        on_time_rate: 92.1,
        performance_score: 96.5,
        specialties: ["Batterie", "Pneus", "Mécanique générale"],
        certifications: [
            {
                name: "Certification Mécanique Auto",
                issuer: "CFPMA",
                date: "2022-06-15",
                verified: true
            }
        ],
        documents: {
            id_card: {status: "verified", date: "2024-02-12"},
            professional_license: {status: "verified", date: "2024-02-12"},
            insurance: {status: "verified", expires: "2025-12-31"}
        },
        vehicle: {
            type: "Moto",
            brand: "Yamaha",
            plate: "CE-4567-MK"
        },
        bank_info: {
            bank_name: "Afriland First Bank",
            account_number: "10005-12345-67890-12",
            verified: true
        }
    };
    
    return `
        <div class="client-details-wrapper">
            <!-- Header -->
            <div class="client-details-header">
                <div class="client-avatar-large">
                    <img src="${getDefaultAvatar(mechanic.name)}" alt="${mechanic.name}">
                    <div class="status-badge status-${mechanic.status}">
                        <i class="bi bi-circle-fill"></i>
                        ${getStatusText(mechanic.status)}
                    </div>
                </div>
                <div class="client-header-info">
                    <h3>${mechanic.name}</h3>
                    <div class="client-meta">
                        <span class="meta-item">
                            <i class="bi bi-shield-check text-success"></i>
                            Vérifié
                        </span>
                        <span class="meta-item">
                            <i class="bi bi-calendar3"></i>
                            Membre depuis ${new Date(mechanic.registration_date).toLocaleDateString('fr-FR')}
                        </span>
                        <span class="meta-item">
                            <i class="bi bi-star-fill text-warning"></i>
                            ${mechanic.rating}/5 (${mechanic.total_reviews} avis)
                        </span>
                    </div>
                </div>
            </div>
            
            <!-- Performance Metrics -->
            <div class="performance-metrics">
                <div class="metric-box">
                    <i class="metric-icon bi bi-tools text-primary"></i>
                    <span class="metric-label">Interventions</span>
                    <div class="metric-value">${mechanic.total_interventions}</div>
                </div>
                <div class="metric-box">
                    <i class="metric-icon bi bi-check-circle text-success"></i>
                    <span class="metric-label">Taux réussite</span>
                    <div class="metric-value">${mechanic.completion_rate}%</div>
                </div>
                <div class="metric-box">
                    <i class="metric-icon bi bi-clock text-info"></i>
                    <span class="metric-label">Temps réponse</span>
                    <div class="metric-value">${mechanic.response_time_avg}</div>
                </div>
                <div class="metric-box">
                    <i class="metric-icon bi bi-currency-dollar text-warning"></i>
                    <span class="metric-label">Revenus totaux</span>
                    <div class="metric-value">${formatCurrency(mechanic.total_earned)}</div>
                </div>
                <div class="metric-box">
                    <i class="metric-icon bi bi-speedometer2 text-gold"></i>
                    <span class="metric-label">Performance</span>
                    <div class="metric-value">${mechanic.performance_score}%</div>
                </div>
            </div>
            
            <!-- Tabs -->
            <ul class="nav nav-tabs client-tabs" role="tablist">
                <li class="nav-item" role="presentation">
                    <button class="nav-link active" data-bs-toggle="tab" data-bs-target="#info-tab">
                        <i class="bi bi-info-circle"></i> Informations
                    </button>
                </li>
                <li class="nav-item" role="presentation">
                    <button class="nav-link" data-bs-toggle="tab" data-bs-target="#certifications-tab">
                        <i class="bi bi-award"></i> Certifications
                    </button>
                </li>
                <li class="nav-item" role="presentation">
                    <button class="nav-link" data-bs-toggle="tab" data-bs-target="#documents-tab">
                        <i class="bi bi-file-earmark-text"></i> Documents
                    </button>
                </li>
                <li class="nav-item" role="presentation">
                    <button class="nav-link" data-bs-toggle="tab" data-bs-target="#finances-tab">
                        <i class="bi bi-wallet2"></i> Finances
                    </button>
                </li>
                <li class="nav-item" role="presentation">
                    <button class="nav-link" data-bs-toggle="tab" data-bs-target="#reviews-tab">
                        <i class="bi bi-star"></i> Avis clients
                    </button>
                </li>
            </ul>
            
            <div class="tab-content client-tab-content">
                <!-- Informations Tab -->
                <div class="tab-pane fade show active" id="info-tab">
                    <div class="info-grid">
                        <div class="info-item">
                            <i class="bi bi-envelope"></i>
                            <div>
                                <span class="info-label">Email</span>
                                <span class="info-value">${mechanic.email}</span>
                            </div>
                        </div>
                        <div class="info-item">
                            <i class="bi bi-phone"></i>
                            <div>
                                <span class="info-label">Téléphone</span>
                                <span class="info-value">${mechanic.phone}</span>
                            </div>
                        </div>
                        <div class="info-item">
                            <i class="bi bi-geo-alt"></i>
                            <div>
                                <span class="info-label">Localisation</span>
                                <span class="info-value">${mechanic.location}</span>
                            </div>
                        </div>
                        <div class="info-item">
                            <i class="bi bi-bounding-box-circles"></i>
                            <div>
                                <span class="info-label">Zone de couverture</span>
                                <span class="info-value">${mechanic.coverage_zone}</span>
                            </div>
                        </div>
                        <div class="info-item">
                            <i class="bi bi-tools"></i>
                            <div>
                                <span class="info-label">Spécialités</span>
                                <span class="info-value">${mechanic.specialties.join(', ')}</span>
                            </div>
                        </div>
                        <div class="info-item">
                            <i class="bi bi-car-front"></i>
                            <div>
                                <span class="info-label">Véhicule</span>
                                <span class="info-value">${mechanic.vehicle.brand} (${mechanic.vehicle.plate})</span>
                            </div>
                        </div>
                    </div>
                </div>
                
                <!-- Certifications Tab -->
                <div class="tab-pane fade" id="certifications-tab">
                    <div class="certifications-list">
                        ${mechanic.certifications.map(cert => `
                            <div class="certification-card">
                                <div class="certification-icon">
                                    <i class="bi bi-award"></i>
                                </div>
                                <div class="certification-info">
                                    <h6 class="certification-name">${cert.name}</h6>
                                    <div class="certification-details">
                                        <span><i class="bi bi-building"></i> ${cert.issuer}</span>
                                        <span class="ms-3"><i class="bi bi-calendar3"></i> ${new Date(cert.date).toLocaleDateString('fr-FR')}</span>
                                    </div>
                                </div>
                                <span class="verification-badge ${cert.verified ? 'verified' : 'pending'}">
                                    <i class="bi bi-${cert.verified ? 'check-circle' : 'clock'}"></i>
                                    ${cert.verified ? 'Vérifié' : 'En attente'}
                                </span>
                            </div>
                        `).join('')}
                    </div>
                </div>
                
                <!-- Documents Tab -->
                <div class="tab-pane fade" id="documents-tab">
                    <div class="documents-grid">
                        ${Object.entries(mechanic.documents).map(([key, doc]) => `
                            <div class="document-card">
                                <div class="document-header">
                                    <div class="document-title">
                                        <i class="bi bi-file-earmark-text"></i>
                                        ${getDocumentName(key)}
                                    </div>
                                    ${getDocumentStatusBadge(doc.status)}
                                </div>
                                <div class="document-body">
                                    ${doc.date ? `
                                        <div class="document-info-item">
                                            <span>Date vérification</span>
                                            <span>${new Date(doc.date).toLocaleDateString('fr-FR')}</span>
                                        </div>
                                    ` : ''}
                                    ${doc.expires ? `
                                        <div class="document-info-item">
                                            <span>Date expiration</span>
                                            <span>${new Date(doc.expires).toLocaleDateString('fr-FR')}</span>
                                        </div>
                                    ` : ''}
                                </div>
                            </div>
                        `).join('')}
                    </div>
                </div>
                
                <!-- Finances Tab -->
                <div class="tab-pane fade" id="finances-tab">
                    <div class="commission-summary">
                        <div class="commission-box total">
                            <i class="bi bi-currency-dollar" style="font-size: 2rem; color: var(--primary-gold);"></i>
                            <div class="commission-amount">${formatCurrency(mechanic.total_earned)}</div>
                            <small class="text-muted">Revenus totaux</small>
                        </div>
                        <div class="commission-box pending">
                            <i class="bi bi-hourglass-split" style="font-size: 2rem; color: #F59E0B;"></i>
                            <div class="commission-amount">${formatCurrency(mechanic.pending_commission)}</div>
                            <small class="text-muted">Commission en attente</small>
                        </div>
                        <div class="commission-box paid">
                            <i class="bi bi-check-circle" style="font-size: 2rem; color: #10B981;"></i>
                            <div class="commission-amount">${formatCurrency(mechanic.available_balance)}</div>
                            <small class="text-muted">Disponible</small>
                        </div>
                    </div>
                    
                    <div class="bank-info-display">
                        <h6 class="mb-3"><i class="bi bi-bank"></i> Informations bancaires</h6>
                        <div class="info-grid">
                            <div class="bank-info-item">
                                <label>Banque</label>
                                <span>${mechanic.bank_info.bank_name}</span>
                            </div>
                            <div class="bank-info-item">
                                <label>Numéro de compte</label>
                                <span>${mechanic.bank_info.account_number}</span>
                            </div>
                            <div class="bank-info-item">
                                <label>Statut</label>
                                <span class="text-${mechanic.bank_info.verified ? 'success' : 'warning'}">
                                    <i class="bi bi-${mechanic.bank_info.verified ? 'check-circle' : 'clock'}"></i>
                                    ${mechanic.bank_info.verified ? 'Vérifié' : 'En attente'}
                                </span>
                            </div>
                        </div>
                    </div>
                </div>
                
                <!-- Reviews Tab -->
                <div class="tab-pane fade" id="reviews-tab">
                    <div class="reviews-list">
                        ${generateMockReviews()}
                    </div>
                </div>
            </div>
            
            <!-- Action Footer -->
            <div class="client-actions-footer">
                <button class="btn btn-outline-secondary" onclick="sendNotification(${mechanicId})">
                    <i class="bi bi-send"></i> Notifier
                </button>
                <button class="btn btn-outline-info" onclick="viewLocation(${mechanicId})">
                    <i class="bi bi-map"></i> Localiser
                </button>
                <button class="btn btn-outline-primary" onclick="editMechanic(${mechanicId})">
                    <i class="bi bi-pencil"></i> Modifier
                </button>
                <button class="btn btn-outline-warning" onclick="suspendMechanic(${mechanicId})">
                    <i class="bi bi-ban"></i> Suspendre
                </button>
            </div>
        </div>
    `;
}

// Helper functions
function getStatusText(status) {
    const texts = {
        'online': 'En ligne',
        'offline': 'Hors ligne',
        'busy': 'Occupé',
        'suspended': 'Suspendu'
    };
    return texts[status] || status;
}

function getDocumentName(key) {
    const names = {
        'id_card': 'Carte d\'identité',
        'professional_license': 'Licence professionnelle',
        'insurance': 'Assurance'
    };
    return names[key] || key;
}

function getDocumentStatusBadge(status) {
    const badges = {
        'verified': '<span class="verification-badge verified"><i class="bi bi-check-circle"></i> Vérifié</span>',
        'pending': '<span class="verification-badge pending"><i class="bi bi-clock"></i> En attente</span>',
        'expired': '<span class="badge bg-danger">Expiré</span>'
    };
    return badges[status] || '';
}

function generateMockReviews() {
    const reviews = [
        {
            author: "Jean Kouam",
            rating: 5,
            comment: "Excellent service ! Très professionnel et rapide. Je recommande vivement.",
            date: "2025-11-15"
        },
        {
            author: "Marie Tchoumi",
            rating: 5,
            comment: "Intervention rapide et efficace. Prix honnête. Merci !",
            date: "2025-11-10"
        },
        {
            author: "Paul Mbarga",
            rating: 4,
            comment: "Bon travail, quelques minutes de retard mais résultat satisfaisant.",
            date: "2025-11-05"
        }
    ];
    
    return reviews.map(review => `
        <div class="review-card">
            <div class="review-header">
                <div class="review-author">
                    <img src="${getDefaultAvatar(review.author)}" alt="${review.author}">
                    <div class="review-author-info">
                        <h6>${review.author}</h6>
                        <small>${new Date(review.date).toLocaleDateString('fr-FR')}</small>
                    </div>
                </div>
                <div class="review-rating">
                    ${Array(5).fill(0).map((_, i) => `
                        <i class="bi bi-star${i < review.rating ? '-fill' : ''}"></i>
                    `).join('')}
                </div>
            </div>
            <p class="review-text">${review.comment}</p>
        </div>
    `).join('');
}

function getDefaultAvatar(name) {
    return `https://ui-avatars.com/api/?name=${encodeURIComponent(name)}&background=E1A004&color=1D1D1B&bold=true&size=128`;
}

/**
 * Autres actions sur les mécaniciens
 */
function editMechanic(mechanicId) {
    console.log('Éditer mécanicien:', mechanicId);
    showToast('Fonctionnalité d\'édition en cours de développement', 'info');
}

function deleteMechanic(mechanicId) {
    if (confirm('Êtes-vous sûr de vouloir supprimer ce mécanicien ?')) {
        toggleLoading(true);
        setTimeout(() => {
            toggleLoading(false);
            document.querySelectorAll('.modal').forEach(modal => {
                const bsModal = bootstrap.Modal.getInstance(modal);
                if (bsModal) bsModal.hide();
            });
            if (typeof dataTable !== 'undefined') {
                dataTable.row(`[data-mechanic-id="${mechanicId}"]`).remove().draw();
            }
            showToast('Mécanicien supprimé avec succès', 'success');
        }, 1000);
    }
}

function suspendMechanic(mechanicId) {
    const reason = prompt('Raison de la suspension :');
    if (reason && reason.trim()) {
        toggleLoading(true);
        setTimeout(() => {
            toggleLoading(false);
            showToast('Mécanicien suspendu avec succès', 'warning');
            document.querySelectorAll('.modal').forEach(modal => {
                const bsModal = bootstrap.Modal.getInstance(modal);
                if (bsModal) bsModal.hide();
            });
            if (typeof dataTable !== 'undefined') {
                dataTable.draw();
            }
        }, 1000);
    }
}

function activateMechanic(mechanicId) {
    toggleLoading(true);
    setTimeout(() => {
        toggleLoading(false);
        showToast('Mécanicien activé avec succès', 'success');
        if (typeof dataTable !== 'undefined') {
            dataTable.draw();
        }
    }, 1000);
}

function viewLocation(mechanicId) {
    console.log('Localiser mécanicien:', mechanicId);
    showToast('Ouverture de la carte...', 'info');
    // TODO: Implémenter vue carte avec localisation
}

function viewDocuments(mechanicId) {
    console.log('Documents du mécanicien:', mechanicId);
    viewMechanic(mechanicId); // Ouvre avec l'onglet documents
}

function viewPerformance(mechanicId) {
    console.log('Performance du mécanicien:', mechanicId);
    showToast('Rapport de performance en cours de développement', 'info');
}

function manageCommissions(mechanicId) {
    console.log('Gérer commissions:', mechanicId);
    viewMechanic(mechanicId); // Ouvre avec l'onglet finances
}

function sendNotification(mechanicId) {
    console.log('Envoyer notification:', mechanicId);
    showToast('Notification envoyée', 'success');
}

// ============================================
// BULK ACTIONS
// ============================================

function bulkAction(action) {
    if (selectedMechanics.length === 0) {
        showToast('Aucun mécanicien sélectionné', 'warning');
        return;
    }
    
    const confirmMessages = {
        verify: `Vérifier ${selectedMechanics.length} mécanicien(s) ?`,
        suspend: `Suspendre ${selectedMechanics.length} mécanicien(s) ?`,
        notify: null,
        export: null
    };
    
    if (confirmMessages[action] && !confirm(confirmMessages[action])) {
        return;
    }
    
    toggleLoading(true);
    
    setTimeout(() => {
        toggleLoading(false);
        
        switch(action) {
            case 'export':
                exportSelectedMechanics();
                break;
            case 'notify':
                showToast(`Notification envoyée à ${selectedMechanics.length} mécanicien(s)`, 'success');
                break;
            default:
                showToast(`Action "${action}" effectuée sur ${selectedMechanics.length} mécanicien(s)`, 'success');
        }
        
        $('.row-checkbox').prop('checked', false);
        $('#selectAll').prop('checked', false);
        selectedMechanics = [];
        $('#bulkActionsBar').slideUp(300);
        
        if (typeof dataTable !== 'undefined') {
            dataTable.draw();
        }
    }, 1500);
}

function exportSelectedMechanics() {
    console.log('Export mécaniciens:', selectedMechanics);
    showToast(`Export de ${selectedMechanics.length} mécanicien(s) en cours...`, 'info');
}

// ============================================
// MAP VIEW
// ============================================

function viewMap() {
    showToast('Carte des mécaniciens en cours de développement', 'info');
    // TODO: Implémenter vue carte avec tous les mécaniciens
}

// ============================================
// EXPORT DATA
// ============================================

function exportData() {
    const buttons = document.querySelector('.dt-buttons');
    if (buttons) {
        buttons.style.display = 'flex';
        buttons.style.gap = '0.5rem';
        buttons.style.marginBottom = '1rem';
        
        setTimeout(() => {
            buttons.style.display = 'none';
        }, 10000);
    }
}