<?php
$pageTitle = 'Gestion des Avis';
$breadcrumbs = [
    ['label' => 'Dashboard', 'url' => '../index.php'],
    ['label' => 'Avis', 'url' => '#']
];

require_once __DIR__ . '/../../config/constants.php';
require_once __DIR__ . '/../../config/functions.php';

// Charger les données mock
$interventionsData = json_decode(file_get_contents(__DIR__ . '/../../mockdata/interventions.json'), true);
$usersData = json_decode(file_get_contents(__DIR__ . '/../../mockdata/users.json'), true);
$mechanicsData = json_decode(file_get_contents(__DIR__ . '/../../mockdata/mechanics.json'), true);

// Générer des avis mock basés sur les interventions terminées
$reviews = [];
$completedInterventions = array_filter($interventionsData['interventions'], function($int) {
    return $int['status'] === 'completed' && isset($int['rating']);
});

foreach($completedInterventions as $intervention) {
    $reviews[] = [
        'id' => 'REV-' . str_pad(rand(1000, 9999), 4, '0', STR_PAD_LEFT),
        'intervention_id' => $intervention['id'],
        'client_id' => $intervention['client_id'],
        'client_name' => $intervention['client'],
        'mechanic_id' => $intervention['mechanic_id'],
        'mechanic_name' => $intervention['mechanic'],
        'rating' => $intervention['rating'],
        'review_text' => $intervention['review'] ?? 'Service rapide et professionnel. Je recommande!',
        'service_type' => $intervention['type'],
        'created_at' => date('Y-m-d H:i:s', strtotime($intervention['completed_at']) + rand(3600, 86400)), // 1h à 24h après completion
        'response' => rand(0, 1) ? 'Merci pour votre retour positif! Nous sommes ravis de vous avoir aidé.' : null,
        'response_at' => rand(0, 1) ? date('Y-m-d H:i:s', strtotime('+2 hours')) : null,
        'status' => 'published',
        'flagged' => rand(0, 10) === 0 // 10% des avis sont signalés
    ];
}

// Ajouter quelques avis supplémentaires avec différentes notes
$additionalReviews = [
    [
        'id' => 'REV-0001',
        'client_name' => 'Marie Tchoumi',
        'mechanic_name' => 'Pierre Nkolo',
        'rating' => 2,
        'review_text' => 'Le mécanicien est arrivé en retard et le travail n\'était pas satisfaisant. La crevaison n\'a pas été correctement réparée.',
        'service_type' => 'Crevaison',
        'created_at' => '2025-11-15 14:30:00',
        'response' => 'Nous nous excusons pour ce désagrément. Nous allons prendre contact avec vous pour arranger cela.',
        'response_at' => '2025-11-15 16:45:00',
        'status' => 'published',
        'flagged' => true
    ],
    [
        'id' => 'REV-0002',
        'client_name' => 'Paul Mbarga',
        'mechanic_name' => 'Albert Manga',
        'rating' => 1,
        'review_text' => 'Service horrible! Le mécanicien n\'est jamais venu malgré plusieurs appels.',
        'service_type' => 'Panne sèche',
        'created_at' => '2025-11-10 09:15:00',
        'response' => null,
        'response_at' => null,
        'status' => 'pending_moderation',
        'flagged' => true
    ]
];

$reviews = array_merge($reviews, $additionalReviews);

// Filtres
$ratingFilter = $_GET['rating'] ?? 'all';
$dateFilter = $_GET['date'] ?? 'all';
$serviceFilter = $_GET['service'] ?? 'all';
$statusFilter = $_GET['status'] ?? 'all';

// Appliquer les filtres
$filteredReviews = array_filter($reviews, function($review) use ($ratingFilter, $dateFilter, $serviceFilter, $statusFilter) {
    if ($ratingFilter !== 'all' && $review['rating'] != $ratingFilter) return false;
    if ($serviceFilter !== 'all' && $review['service_type'] !== $serviceFilter) return false;
    if ($statusFilter !== 'all' && $review['status'] !== $statusFilter) return false;
    if ($dateFilter === 'week' && strtotime($review['created_at']) < strtotime('-1 week')) return false;
    if ($dateFilter === 'month' && strtotime($review['created_at']) < strtotime('-1 month')) return false;
    return true;
});

// Trier par date décroissante
usort($filteredReviews, function($a, $b) {
    return strtotime($b['created_at']) - strtotime($a['created_at']);
});

// Pagination
$page = $_GET['page'] ?? 1;
$perPage = 10;
$totalPages = ceil(count($filteredReviews) / $perPage);
$offset = ($page - 1) * $perPage;
$pagedReviews = array_slice($filteredReviews, $offset, $perPage);

// Statistiques
$stats = [
    'total_reviews' => count($reviews),
    'avg_rating' => number_format(array_sum(array_column($reviews, 'rating')) / count($reviews), 1),
    'five_star' => count(array_filter($reviews, fn($r) => $r['rating'] == 5)),
    'four_star' => count(array_filter($reviews, fn($r) => $r['rating'] == 4)),
    'three_star' => count(array_filter($reviews, fn($r) => $r['rating'] == 3)),
    'two_star' => count(array_filter($reviews, fn($r) => $r['rating'] == 2)),
    'one_star' => count(array_filter($reviews, fn($r) => $r['rating'] == 1)),
    'flagged' => count(array_filter($reviews, fn($r) => $r['flagged'])),
    'responded' => count(array_filter($reviews, fn($r) => $r['response'] !== null))
];
?>

<!-- Page Header -->
<div class="page-header">
    <div class="page-header-content">
        <h1 class="page-title">
            <i class="bi bi-star"></i>
            Gestion des Avis
        </h1>
        <p class="page-subtitle">Modérer et répondre aux avis clients</p>
    </div>
    <div class="page-header-actions">
        <div class="rating-overview">
            <div class="rating-display">
                <div class="rating-stars">
                    <?php for($i = 1; $i <= 5; $i++): ?>
                        <i class="bi bi-star<?= $i <= $stats['avg_rating'] ? '-fill' : '' ?> text-warning"></i>
                    <?php endfor; ?>
                </div>
                <div class="rating-score">
                    <span class="score-value"><?= $stats['avg_rating'] ?></span>
                    <span class="score-total">/5</span>
                </div>
            </div>
            <small class="text-muted"><?= $stats['total_reviews'] ?> avis</small>
        </div>
    </div>
</div>

<!-- Rating Analytics -->
<div class="row g-4 mb-4">
    <div class="col-xl-8">
        <div class="dashboard-card">
            <div class="card-header">
                <h5 class="card-title">
                    <i class="bi bi-bar-chart"></i>
                    Répartition des Notes
                </h5>
            </div>
            <div class="card-body">
                <div class="rating-breakdown">
                    <?php for($i = 5; $i >= 1; $i--): ?>
                    <div class="rating-row">
                        <div class="rating-label">
                            <span class="rating-number"><?= $i ?></span>
                            <i class="bi bi-star-fill text-warning"></i>
                        </div>
                        <div class="rating-bar">
                            <div class="rating-fill" style="width: <?= ($stats[$i == 1 ? 'one' : ($i == 2 ? 'two' : ($i == 3 ? 'three' : ($i == 4 ? 'four' : 'five'))).'_star'] / $stats['total_reviews']) * 100 ?>%"></div>
                        </div>
                        <div class="rating-count">
                            <span class="count-value"><?= $stats[$i == 1 ? 'one' : ($i == 2 ? 'two' : ($i == 3 ? 'three' : ($i == 4 ? 'four' : 'five'))).'_star'] ?></span>
                            <small class="text-muted">(<?= number_format(($stats[$i == 1 ? 'one' : ($i == 2 ? 'two' : ($i == 3 ? 'three' : ($i == 4 ? 'four' : 'five'))).'_star'] / $stats['total_reviews']) * 100, 1) ?>%)</small>
                        </div>
                    </div>
                    <?php endfor; ?>
                </div>
            </div>
        </div>
    </div>

    <div class="col-xl-4">
        <div class="dashboard-card">
            <div class="card-header">
                <h5 class="card-title">
                    <i class="bi bi-graph-up"></i>
                    Statistiques
                </h5>
            </div>
            <div class="card-body">
                <div class="stats-grid">
                    <div class="stat-item">
                        <div class="stat-icon bg-success">
                            <i class="bi bi-chat-quote"></i>
                        </div>
                        <div class="stat-content">
                            <span class="stat-value"><?= $stats['responded'] ?></span>
                            <span class="stat-label">Réponses</span>
                        </div>
                    </div>

                    <div class="stat-item">
                        <div class="stat-icon bg-warning">
                            <i class="bi bi-flag"></i>
                        </div>
                        <div class="stat-content">
                            <span class="stat-value"><?= $stats['flagged'] ?></span>
                            <span class="stat-label">Signalés</span>
                        </div>
                    </div>

                    <div class="stat-item">
                        <div class="stat-icon bg-info">
                            <i class="bi bi-clock"></i>
                        </div>
                        <div class="stat-content">
                            <span class="stat-value">2.4h</span>
                            <span class="stat-label">Temps moyen réponse</span>
                        </div>
                    </div>

                    <div class="stat-item">
                        <div class="stat-icon bg-primary">
                            <i class="bi bi-trending-up"></i>
                        </div>
                        <div class="stat-content">
                            <span class="stat-value">+0.2</span>
                            <span class="stat-label">Évolution mensuelle</span>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>

<!-- Filters Bar -->
<div class="dashboard-card mb-4">
    <div class="card-body">
        <form method="GET" class="filters-form">
            <div class="row g-3">
                <div class="col-md-2">
                    <label class="form-label">Note</label>
                    <select name="rating" class="form-select">
                        <option value="all" <?= $ratingFilter === 'all' ? 'selected' : '' ?>>Toutes</option>
                        <option value="5" <?= $ratingFilter === '5' ? 'selected' : '' ?>>⭐⭐⭐⭐⭐ 5 étoiles</option>
                        <option value="4" <?= $ratingFilter === '4' ? 'selected' : '' ?>>⭐⭐⭐⭐ 4 étoiles</option>
                        <option value="3" <?= $ratingFilter === '3' ? 'selected' : '' ?>>⭐⭐⭐ 3 étoiles</option>
                        <option value="2" <?= $ratingFilter === '2' ? 'selected' : '' ?>>⭐⭐ 2 étoiles</option>
                        <option value="1" <?= $ratingFilter === '1' ? 'selected' : '' ?>>⭐ 1 étoile</option>
                    </select>
                </div>

                <div class="col-md-2">
                    <label class="form-label">Service</label>
                    <select name="service" class="form-select">
                        <option value="all" <?= $serviceFilter === 'all' ? 'selected' : '' ?>>Tous</option>
                        <option value="Batterie" <?= $serviceFilter === 'Batterie' ? 'selected' : '' ?>>Batterie</option>
                        <option value="Crevaison" <?= $serviceFilter === 'Crevaison' ? 'selected' : '' ?>>Crevaison</option>
                        <option value="Panne sèche" <?= $serviceFilter === 'Panne sèche' ? 'selected' : '' ?>>Panne sèche</option>
                        <option value="Mécanique générale" <?= $serviceFilter === 'Mécanique générale' ? 'selected' : '' ?>>Mécanique générale</option>
                        <option value="Transmission" <?= $serviceFilter === 'Transmission' ? 'selected' : '' ?>>Transmission</option>
                        <option value="Climatisation" <?= $serviceFilter === 'Climatisation' ? 'selected' : '' ?>>Climatisation</option>
                    </select>
                </div>

                <div class="col-md-2">
                    <label class="form-label">Période</label>
                    <select name="date" class="form-select">
                        <option value="all" <?= $dateFilter === 'all' ? 'selected' : '' ?>>Toutes</option>
                        <option value="week" <?= $dateFilter === 'week' ? 'selected' : '' ?>>Cette semaine</option>
                        <option value="month" <?= $dateFilter === 'month' ? 'selected' : '' ?>>Ce mois</option>
                    </select>
                </div>

                <div class="col-md-2">
                    <label class="form-label">Statut</label>
                    <select name="status" class="form-select">
                        <option value="all" <?= $statusFilter === 'all' ? 'selected' : '' ?>>Tous</option>
                        <option value="published" <?= $statusFilter === 'published' ? 'selected' : '' ?>>Publiés</option>
                        <option value="pending_moderation" <?= $statusFilter === 'pending_moderation' ? 'selected' : '' ?>>En attente</option>
                        <option value="hidden" <?= $statusFilter === 'hidden' ? 'selected' : '' ?>>Masqués</option>
                    </select>
                </div>

                <div class="col-md-4">
                    <label class="form-label">&nbsp;</label>
                    <div class="d-flex gap-2">
                        <button type="submit" class="btn btn-primary">
                            <i class="bi bi-search"></i>
                            Filtrer
                        </button>
                        <a href="reviews.php" class="btn btn-outline-secondary">
                            <i class="bi bi-x"></i>
                            Réinitialiser
                        </a>
                    </div>
                </div>
            </div>
        </form>
    </div>
</div>

<!-- Reviews List -->
<div class="reviews-list">
    <?php foreach($pagedReviews as $review): ?>
    <div class="review-card <?= $review['flagged'] ? 'flagged' : '' ?> <?= $review['status'] === 'pending_moderation' ? 'pending' : '' ?>">
        <div class="review-header">
            <div class="review-user">
                <img src="<?= getDefaultAvatar($review['client_name']) ?>" alt="" class="user-avatar">
                <div class="user-info">
                    <span class="user-name"><?= e($review['client_name']) ?></span>
                    <div class="review-meta">
                        <span class="service-type"><?= e($review['service_type']) ?></span>
                        <span class="review-date">
                            <i class="bi bi-clock"></i>
                            <?= timeAgo($review['created_at']) ?>
                        </span>
                    </div>
                </div>
            </div>
            <div class="review-rating">
                <div class="rating-stars">
                    <?php for($i = 1; $i <= 5; $i++): ?>
                        <i class="bi bi-star<?= $i <= $review['rating'] ? '-fill' : '' ?> <?= $review['rating'] >= $i ? 'active' : '' ?>"></i>
                    <?php endfor; ?>
                </div>
                <span class="rating-number"><?= $review['rating'] ?>/5</span>
            </div>
        </div>

        <div class="review-content">
            <p class="review-text"><?= e($review['review_text']) ?></p>
        </div>

        <?php if($review['response']): ?>
        <div class="review-response">
            <div class="response-header">
                <span class="response-label">Réponse de <?= e($review['mechanic_name']) ?></span>
                <small class="response-date">
                    <i class="bi bi-clock"></i>
                    <?= timeAgo($review['response_at']) ?>
                </small>
            </div>
            <p class="response-text"><?= e($review['response']) ?></p>
        </div>
        <?php endif; ?>

        <div class="review-actions">
            <div class="action-buttons">
                <?php if(!$review['response']): ?>
                <button class="btn btn-sm btn-outline-primary" onclick="replyToReview('<?= $review['id'] ?>')">
                    <i class="bi bi-reply"></i>
                    Répondre
                </button>
                <?php endif; ?>

                <button class="btn btn-sm btn-outline-warning" onclick="flagReview('<?= $review['id'] ?>')">
                    <i class="bi bi-flag"></i>
                    Signaler
                </button>

                <button class="btn btn-sm btn-outline-danger" onclick="deleteReview('<?= $review['id'] ?>')">
                    <i class="bi bi-trash"></i>
                    Supprimer
                </button>
            </div>

            <?php if($review['flagged']): ?>
            <div class="review-flags">
                <span class="flag-badge">
                    <i class="bi bi-exclamation-triangle"></i>
                    Signalé
                </span>
            </div>
            <?php endif; ?>

            <?php if($review['status'] === 'pending_moderation'): ?>
            <div class="moderation-actions">
                <button class="btn btn-sm btn-success" onclick="approveReview('<?= $review['id'] ?>')">
                    <i class="bi bi-check"></i>
                    Approuver
                </button>
                <button class="btn btn-sm btn-danger" onclick="rejectReview('<?= $review['id'] ?>')">
                    <i class="bi bi-x"></i>
                    Rejeter
                </button>
            </div>
            <?php endif; ?>
        </div>
    </div>
    <?php endforeach; ?>
</div>

<!-- Pagination -->
<?php if($totalPages > 1): ?>
<div class="d-flex justify-content-center mt-4">
    <nav aria-label="Pagination">
        <ul class="pagination">
            <li class="page-item <?= $page <= 1 ? 'disabled' : '' ?>">
                <a class="page-link" href="?page=<?= $page - 1 ?>&rating=<?= $ratingFilter ?>&service=<?= $serviceFilter ?>&date=<?= $dateFilter ?>&status=<?= $statusFilter ?>">
                    <i class="bi bi-chevron-left"></i>
                </a>
            </li>

            <?php for($i = max(1, $page - 2); $i <= min($totalPages, $page + 2); $i++): ?>
            <li class="page-item <?= $i === $page ? 'active' : '' ?>">
                <a class="page-link" href="?page=<?= $i ?>&rating=<?= $ratingFilter ?>&service=<?= $serviceFilter ?>&date=<?= $dateFilter ?>&status=<?= $statusFilter ?>"><?= $i ?></a>
            </li>
            <?php endfor; ?>

            <li class="page-item <?= $page >= $totalPages ? 'disabled' : '' ?>">
                <a class="page-link" href="?page=<?= $page + 1 ?>&rating=<?= $ratingFilter ?>&service=<?= $serviceFilter ?>&date=<?= $dateFilter ?>&status=<?= $statusFilter ?>">
                    <i class="bi bi-chevron-right"></i>
                </a>
            </li>
        </ul>
    </nav>
</div>
<?php endif; ?>

<!-- Modal: Reply to Review -->
<div class="modal fade" id="replyModal" tabindex="-1">
    <div class="modal-dialog">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title">
                    <i class="bi bi-reply"></i>
                    Répondre à l'avis
                </h5>
                <button type="button" class="btn-close btn-close-white" data-bs-dismiss="modal"></button>
            </div>
            <form id="replyForm">
                <div class="modal-body">
                    <div class="mb-3">
                        <label class="form-label">Votre réponse</label>
                        <textarea class="form-control" name="response" rows="4" placeholder="Tapez votre réponse..." required></textarea>
                        <small class="text-muted">Cette réponse sera visible par le client</small>
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Annuler</button>
                    <button type="submit" class="btn btn-primary">
                        <i class="bi bi-send"></i>
                        Envoyer la réponse
                    </button>
                </div>
            </form>
        </div>
    </div>
</div>

<script>
// ========================================
// REVIEWS FUNCTIONALITY
// ========================================
document.addEventListener('DOMContentLoaded', function() {
    // Reply form submission
    document.getElementById('replyForm').addEventListener('submit', function(e) {
        e.preventDefault();
        const response = this.querySelector('[name="response"]').value;

        if(response.trim()) {
            showNotification('Réponse envoyée avec succès!', 'success');
            const modal = bootstrap.Modal.getInstance(document.getElementById('replyModal'));
            modal.hide();
            this.reset();
        }
    });
});

// Functions
function replyToReview(reviewId) {
    const modal = new bootstrap.Modal(document.getElementById('replyModal'));
    modal.show();
}

function flagReview(reviewId) {
    if(confirm('Êtes-vous sûr de vouloir signaler cet avis?')) {
        showNotification('Avis signalé pour modération', 'warning');
    }
}

function deleteReview(reviewId) {
    if(confirm('Êtes-vous sûr de vouloir supprimer cet avis? Cette action est irréversible.')) {
        showNotification('Avis supprimé', 'danger');
    }
}

function approveReview(reviewId) {
    showNotification('Avis approuvé et publié', 'success');
}

function rejectReview(reviewId) {
    showNotification('Avis rejeté', 'warning');
}

function showNotification(message, type = 'info') {
    const notification = document.createElement('div');
    notification.className = `alert alert-${type} alert-dismissible fade show position-fixed`;
    notification.style.cssText = 'top: 20px; right: 20px; z-index: 9999; min-width: 300px;';
    notification.innerHTML = `
        ${message}
        <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
    `;

    document.body.appendChild(notification);

    setTimeout(() => {
        if(notification.parentNode) {
            notification.remove();
        }
    }, 5000);
}
</script>

<style>
/* Rating Overview */
.rating-overview {
    text-align: center;
}

.rating-display {
    display: flex;
    align-items: center;
    justify-content: center;
    gap: 12px;
    margin-bottom: 4px;
}

.rating-stars {
    display: flex;
    gap: 2px;
}

.rating-stars i {
    font-size: 16px;
}

.rating-score {
    display: flex;
    align-items: baseline;
    gap: 2px;
}

.score-value {
    font-size: 24px;
    font-weight: 700;
    color: var(--text-primary);
}

.score-total {
    font-size: 14px;
    color: var(--text-secondary);
}

/* Rating Breakdown */
.rating-breakdown {
    display: flex;
    flex-direction: column;
    gap: 12px;
}

.rating-row {
    display: flex;
    align-items: center;
    gap: 12px;
}

.rating-label {
    display: flex;
    align-items: center;
    gap: 8px;
    min-width: 60px;
}

.rating-number {
    font-weight: 600;
    color: var(--text-primary);
}

.rating-bar {
    flex: 1;
    height: 8px;
    background: rgba(255, 255, 255, 0.1);
    border-radius: 4px;
    overflow: hidden;
}

.rating-fill {
    height: 100%;
    background: var(--primary-color);
    border-radius: 4px;
}

.rating-count {
    min-width: 80px;
    text-align: right;
}

.count-value {
    font-weight: 600;
    color: var(--text-primary);
}

/* Stats Grid */
.stats-grid {
    display: grid;
    grid-template-columns: 1fr 1fr;
    gap: 16px;
}

.stat-item {
    display: flex;
    align-items: center;
    gap: 12px;
    padding: 12px;
    background: rgba(255, 255, 255, 0.02);
    border-radius: 8px;
}

.stat-icon {
    width: 40px;
    height: 40px;
    border-radius: 8px;
    display: flex;
    align-items: center;
    justify-content: center;
    color: white;
}

.stat-content {
    display: flex;
    flex-direction: column;
}

.stat-value {
    font-size: 18px;
    font-weight: 700;
    color: var(--text-primary);
}

.stat-label {
    font-size: 12px;
    color: var(--text-secondary);
}

/* Reviews List */
.reviews-list {
    display: flex;
    flex-direction: column;
    gap: 16px;
}

.review-card {
    background: var(--card-background);
    border: 1px solid var(--border-color);
    border-radius: 12px;
    padding: 20px;
    transition: all 0.2s ease;
}

.review-card:hover {
    box-shadow: 0 4px 20px rgba(0, 0, 0, 0.3);
}

.review-card.flagged {
    border-color: #F59E0B;
    background: rgba(245, 158, 11, 0.05);
}

.review-card.pending {
    border-color: #3B82F6;
    background: rgba(59, 130, 246, 0.05);
}

.review-header {
    display: flex;
    justify-content: space-between;
    align-items: flex-start;
    margin-bottom: 16px;
}

.review-user {
    display: flex;
    align-items: center;
    gap: 12px;
}

.user-avatar {
    width: 48px;
    height: 48px;
    border-radius: 50%;
    object-fit: cover;
}

.user-info {
    display: flex;
    flex-direction: column;
}

.user-name {
    font-weight: 600;
    color: var(--text-primary);
    margin-bottom: 4px;
}

.review-meta {
    display: flex;
    gap: 12px;
    font-size: 12px;
    color: var(--text-secondary);
}

.review-rating {
    display: flex;
    flex-direction: column;
    align-items: flex-end;
    gap: 4px;
}

.rating-stars {
    display: flex;
    gap: 2px;
}

.rating-stars i {
    font-size: 14px;
    color: #6B7280;
}

.rating-stars i.active {
    color: #F59E0B;
}

.rating-number {
    font-size: 12px;
    color: var(--text-secondary);
}

.review-content {
    margin-bottom: 16px;
}

.review-text {
    margin: 0;
    color: var(--text-primary);
    line-height: 1.5;
}

.review-response {
    background: rgba(16, 185, 129, 0.1);
    border: 1px solid rgba(16, 185, 129, 0.2);
    border-radius: 8px;
    padding: 16px;
    margin-bottom: 16px;
}

.response-header {
    display: flex;
    justify-content: space-between;
    align-items: center;
    margin-bottom: 8px;
}

.response-label {
    font-weight: 600;
    color: #10B981;
}

.response-date {
    font-size: 12px;
    color: var(--text-secondary);
}

.response-text {
    margin: 0;
    color: var(--text-primary);
    font-style: italic;
}

.review-actions {
    display: flex;
    justify-content: space-between;
    align-items: center;
}

.action-buttons {
    display: flex;
    gap: 8px;
}

.review-flags, .moderation-actions {
    display: flex;
    gap: 8px;
}

.flag-badge {
    display: inline-flex;
    align-items: center;
    gap: 4px;
    background: rgba(245, 158, 11, 0.1);
    color: #F59E0B;
    padding: 4px 8px;
    border-radius: 4px;
    font-size: 12px;
    font-weight: 500;
}

/* Pagination */
.pagination {
    margin-bottom: 0;
}

.pagination .page-link {
    background: var(--card-background);
    border-color: var(--border-color);
    color: var(--text-primary);
    padding: 8px 12px;
}

.pagination .page-link:hover {
    background: rgba(255, 255, 255, 0.05);
    border-color: var(--primary-color);
    color: var(--primary-color);
}

.pagination .page-item.active .page-link {
    background: var(--primary-color);
    border-color: var(--primary-color);
}

/* Modal Styles */
.modal-content {
    background: var(--card-background);
    border: 1px solid var(--border-color);
}

.modal-header {
    border-bottom: 1px solid var(--border-color);
}

.modal-footer {
    border-top: 1px solid var(--border-color);
}

/* Responsive */
@media (max-width: 768px) {
    .rating-display {
        flex-direction: column;
        gap: 8px;
    }

    .rating-breakdown {
        gap: 8px;
    }

    .rating-row {
        gap: 8px;
    }

    .stats-grid {
        grid-template-columns: 1fr;
    }

    .review-header {
        flex-direction: column;
        gap: 12px;
    }

    .review-rating {
        align-items: flex-start;
    }

    .review-actions {
        flex-direction: column;
        gap: 12px;
    }

    .action-buttons {
        flex-wrap: wrap;
    }
}
</style>
